<?php

namespace App\Models;

use App\Models\Login;
use Illuminate\Database\Eloquent\Model;
use Laravel\Sanctum\Contracts\HasAbilities;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class PersonalAccessToken extends Model implements HasAbilities
{
    use HasFactory;

    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
    protected $casts = [
        'abilities' => 'json',
        'access_token_expired_at' => 'datetime',
        'refresh_token_expired_at' => 'datetime',
        'last_used_at' => 'datetime',
    ];

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'device_name',
        'token',
        'access_token',
        'access_token_expired_at',
        'refresh_token',
        'refresh_token_expired_at',
        'device_type',
        'device_id',
        'ip',
        'fcm_key',
        'abilities',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array
     */
    protected $hidden = [
        'access_token',
        'refresh_token',
    ];

    protected function deviceType(): Attribute
    {
        return Attribute::make(
            set: fn ($value) => $this->deviceTypeIdentifier($value),
        );
    }

    /**
     * Get the tokenable model that the access token belongs to.
     *
     * @return \Illuminate\Database\Eloquent\Relations\MorphTo
     */
    public function tokenable()
    {
        return $this->morphTo('tokenable');
    }

    /**
     * Find the token instance matching the given token.
     *
     * @param  string  $token
     * @return static|null
     */
    public static function findToken($token)
    {
        if (strpos($token, '|') === false) {
            return static::where('access_token', hash('sha256', $token))->first();
        }

        [$id, $token] = explode('|', $token, 2);
        if ($instance = static::find($id)) {
            return hash_equals($instance->access_token, hash('sha256', $token)) ? $instance : null;
        }
    }

    /**
     * Find the token instance matching the given token.
     *
     * @param  string  $token
     * @return static|null
     */
    public static function findRefreshToken($token)
    {
        if (strpos($token, '|') === false) {
            return static::where('refresh_token', hash('sha256', $token))->first();
        }

        [$id, $token] = explode('|', $token, 2);
        if ($instance = static::find($id)) {
            return hash_equals($instance->refresh_token, hash('sha256', $token)) ? $instance : null;
        }
    }

    /**
     * Determine if the token has a given ability.
     *
     * @param  string  $ability
     * @return bool
     */
    public function can($ability)
    {
        return in_array('*', $this->abilities) ||
               array_key_exists($ability, array_flip($this->abilities));
    }

    /**
     * Determine if the token is missing a given ability.
     *
     * @param  string  $ability
     * @return bool
     */
    public function cant($ability)
    {
        return ! $this->can($ability);
    }

    public function deviceTypeIdentifier(string $type)
    {
        return match($type) {
            'ios' => 1,
            'android' => 2,
            'web' => 3
        };
    }
}
