<?php

namespace App\Models;

use Illuminate\Support\Str;
use Laravel\Sanctum\HasApiTokens;
use Illuminate\Notifications\Notifiable;
use App\Models\Login;
use App\Models\SocialLogin;
use App\Notifications\AppResetPasswordNotification;
use SpaceO\RESTAuth\Services\NewAccessToken;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\Relations\MorphOne;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Laravel\Sanctum\Sanctum;

class Driver extends Authenticatable
{
    use HasApiTokens, HasFactory, Notifiable;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'uuid',
        'email',
        'full_name',
        'password',
        'is_active',
        'isd_code',
        'mobile',
        'otp',
        'otp_expiry_date',
        'profile_image',
        'stripe_id',
        'driver_customer_stripe_id',
        'sendbird_chat_id',
        'step',
        'is_approved',
        'license_front_image',
        'license_back_image',
        'profile_image',
        'reason',
        'is_active'
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    // protected $casts = [
    //     'email_verified_at' => 'datetime',
    // ];

    // protected $dates = [
    //     'otp_expired_at',
    //     'mobile_verified_at',
    //     'email_otp_expired_at',
    //     'mobile_otp_expired_at',
    // ];

    protected $attributes = [
        // 'locale' => 'en',
        'is_active' => 1,
        'push_notification_accept' => 0,
        'online_status' => 0,
        'is_profile_completed' => 0,
    ];

    public function createToken(array $data, $abilities = ['*'])
    {

        $token = $this->tokens()->create([
            'device_name' => $data['device_name'],
            'access_token' => hash('sha256', $plainTextToken = Str::random(40)),
            'access_token_expired_at' => config('sanctum.expiration') ? now()->addMinutes(config('sanctum.expiration')) : null,
            'refresh_token' => hash('sha256', $plainTextRefreshToken = Str::random(40)),
            'refresh_token_expired_at' => config('rest_auth.refresh_token_life') ? now()->addMinutes(config('rest_auth.refresh_token_life')) : null,
            'device_type' => $data['device_type'],
            'device_id' => $data['device_id'],
            'ip' => request()->ip(),
            'fcm_key' => $data['device_token'] ?? null,
            'abilities' => $abilities,
        ]);

        return new NewAccessToken(
            $token,
            $token->getKey() . '|' . $plainTextToken,
            $token->getKey() . '|' . $plainTextRefreshToken
        );
    }

    public function sendPasswordResetNotification($token)
    {
        $this->notify(new AppResetPasswordNotification($token, 2));
    }

    public function vehicleinfo()
    {
        return $this->hasOne(VehicleInfo::class);
    }

    /**
     * Get the driver's device.
     */
    public function device(): MorphOne
    {
        return $this->morphOne(PersonalAccessToken::class, 'tokenable');
    }

    public function review(): MorphMany
    {
        return $this->morphMany(Review::class, 'reviewable');
    }

    public function isDriver()
    {
        return true; // Override this in the Driver model
    }

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($model) {
            // Set UUID before creating a new model
            $model->uuid = GetUuid();
        });
    }

    public function banks()
    {
        return $this->hasMany(BankAccount::class);
    }
    public function bookings()
    {
        return $this->hasMany(Booking::class);
    }

    public function routeNotificationForFcm()
    {
        if (!empty($this->personalAccess->fcm_key) && !is_null($this->personalAccess->fcm_key)) {
            return $this->personalAccess->fcm_key;
        }
    }

    public function personalAccess()
    {
        return $this->morphOne(Sanctum::$personalAccessTokenModel, 'tokenable');
    }

    public function usersubscriptions() {
        return $this->morphMany(UserSubscription::class, 'subscribable');
    }
}
