<?php

namespace App\Http\Requests;

use Illuminate\Validation\Rule;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Http\Exceptions\HttpResponseException;

class VerifyOTPRequest extends FormRequest
{
    private int $otpLength;

    public function __construct()
    {
        $this->otpLength = config('rest_auth.otp_length');
    }

    public function rules(): array
    {
        $userType = $this->input('user_type');
        return [
            'verification_for' => [
                'required',
                'in:email,mobile'
            ],
            'email' => [
                Rule::requiredIf(function () {
                    return $this->verification_for === 'email';
                }),
                Rule::when(function () {
                    return $this->verification_for === 'email';
                }, [
                    'required',
                    'email',

                    Rule::unique('users')->ignore(auth()->user()),
                    function ($attribute, $value, $fail) {
                        if (
                            auth()->user()->email === $value &&
                            !is_null(auth()->user()->email_verified_at)
                        ) {
                            $fail(__('rest-auth::auth.email_already_yours'));
                        }
                    }
                ]),
            ],
            'isd_code' => [
                'required_if:verification_for,mobile'
            ],
            $userTypeRules = ($userType === '2') ? [
                'mobile' => [
                    'required_if:verification_for,mobile',
                    Rule::unique('drivers', 'mobile')->ignore(auth()->id()),
                    function ($attribute, $value, $fail) {
                        if (
                            auth()->user()->mobile === $value &&
                            auth()->user()->isd_code === $this->isd_code &&
                            !is_null(auth()->user()->mobile_verified_at)
                        ) {
                            $fail(__('rest-auth::auth.mobile_already_yours'));
                        }
                    },
                ],
            ] : [
                'mobile' => [
                    'required_if:verification_for,mobile',
                    Rule::unique('users', 'mobile')->ignore(auth()->id()),
                    function ($attribute, $value, $fail) {

                        if (
                            auth()->user()->mobile === $value &&
                            auth()->user()->isd_code === $this->isd_code &&
                            !is_null(auth()->user()->mobile_verified_at)
                        ) {
                            $fail(__('rest-auth::auth.mobile_already_yours'));
                        }
                    }
                ]
            ],
            'otp' => [
                'required',
                'numeric',
                "digits:{$this->otpLength}",
                function ($attribute, $value, $fail) {
                    if (
                        ($this->verification_for === 'email'
                            && optional(auth()->user()->email_otp_expired_at)->isPast()
                        ) ||
                        ($this->verification_for === 'mobile'
                            && optional(auth()->user()->mobile_otp_expired_at)->isPast()
                        )
                    ) {
                        $fail(__('rest-auth::auth.otp.expired'));
                    }
                },
                function ($attribute, $value, $fail) {

                    if (
                        (
                            ($this->verification_for === 'email' && auth()->user()->email_otp != $value)
                            || ($this->verification_for === 'mobile' && auth()->user()->mobile_otp != $value)
                        )
                        && !($value == 123456 || $value == 1234)
                    ) {
                        $fail(__('rest-auth::auth.otp.not_matched'));
                    }
                }
            ]
        ];
    }

    public function messages(): array
    {
        return [];
    }

    protected function prepareForValidation()
    {
        if ($this->filled('email')) {
            $this->email = strtolower($this->email);
        }
    }

    protected function failedValidation(Validator $validator)
    {
        throw new HttpResponseException(
            response([
                'message' => $validator->errors()->first()
            ], 422)
        );
    }
}
