<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Http\Exceptions\HttpResponseException;

class SendOTPRequest extends FormRequest
{
    public function rules(): array
    {
        $userType = $this->input('user_type');
        return [
            'verification_for' => [
                'required',
                'in:email,mobile'
            ],
            'email' => [
                Rule::requiredIf(function () {
                    return $this->verification_for === 'email';
                }),
                Rule::when(function () {
                    return $this->verification_for === 'email';
                }, [
                    'required',
                    'email',

                    Rule::unique('users')->ignore(auth()->user()),
                    function ($attribute, $value, $fail) {
                        if (
                            auth()->user()->email === $value &&
                            !is_null(auth()->user()->email_verified_at)
                        ) {
                            $fail(__('rest-auth::auth.email_already_yours'));
                        }
                    }
                ]),
            ],
            'isd_code' => [
                'required_if:verification_for,mobile'
            ],
            $userTypeRules = ($userType === '2') ? [
                'mobile' => [
                    'required_if:verification_for,mobile',
                    Rule::unique('drivers', 'mobile')->ignore(auth()->id()),
                    function ($attribute, $value, $fail) {
                        if (
                            auth()->user()->mobile === $value &&
                            auth()->user()->isd_code === $this->isd_code &&
                            !is_null(auth()->user()->mobile_verified_at)
                        ) {
                            $fail(__('rest-auth::auth.mobile_already_yours'));
                        }
                    },
                ],
            ] : [
                'mobile' => [
                    'required_if:verification_for,mobile',
                    Rule::unique('users', 'mobile')->ignore(auth()->id()),
                    function ($attribute, $value, $fail) {

                        if (
                            auth()->user()->mobile === $value &&
                            auth()->user()->isd_code === $this->isd_code &&
                            !is_null(auth()->user()->mobile_verified_at)
                        ) {
                            $fail(__('rest-auth::auth.mobile_already_yours'));
                        }
                    }
                ]
                ]
        ];
    }

    public function messages(): array
    {
        return [];
    }

    protected function prepareForValidation()
    {
        if ($this->filled('email')) {
            $this->email = strtolower($this->email);
        }
    }

    protected function failedValidation(Validator $validator)
    {
        throw new HttpResponseException(
            response([
                'message' => $validator->errors()->first()
            ], 422)
        );
    }
}
