<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;
use Illuminate\Validation\Rules\Password;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Http\Exceptions\HttpResponseException;

class DriverRegisterRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array|string>
     */
    public function rules(): array
    {

        return [
            'full_name' => [
                'required',
                'string',
                'min:3',
                'max:255'
            ],
            'email' => [
                'required',
                'string',
                'email',
                'max:255',
                Rule::unique('drivers', 'email')
            ],
            'password' => [
                'required',
                Password::min(8)
                    ->mixedCase()
                    ->letters()
                    ->numbers()
                    ->symbols()
                    ->uncompromised(),
            ],
            'isd_code' => [
                'required'
            ],
            'mobile' => [
                'required',
                Rule::unique('drivers', 'mobile'),
            ],
            'usermobile' => [
                'phone:AUTO',
            ],
            'device_name' => [
                'required',
                'max:255'
            ],
            'device_type' => [
                'required',
                'in:ios,android,web'
            ],
            'device_id' => [
                'required',
                'max:255'
            ]
        ];
    }

    public function messages(): array
    {
        return [
            'usermobile.phone' => __('rest-auth::validation.phone')
        ];
    }

    protected function prepareForValidation()
    {
        $this->email = strtolower($this->email);
        // Assuming 'isd_code' and 'mobile' are the names of the fields in the form
        $isdCode = $this->input('isd_code');
        $mobile = $this->input('mobile');

        // Modify the 'mobile' field by combining 'isd_code' and 'mobile'
        $this->merge([
            'usermobile' => $isdCode . $mobile,
        ]);
    }

    protected function failedValidation(Validator $validator)
    {
        throw new HttpResponseException(
            response([
                'message' => $validator->errors()->first()
            ], 422)
        );
    }
}
