<?php

namespace App\Http\Controllers\AirTransport;

use App\Models\Terminal;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Http\Controllers\Controller;
use App\Http\Requests\TerminalRequest;
use App\Models\TerminalImage;
use Carbon\Carbon;
use Illuminate\Support\Facades\Storage;
use Inertia\Inertia;
use Yajra\DataTables\Facades\DataTables;

use function Termwind\terminal;

class TerminalController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $dataTable = [];
        if ($request->ajax()) {
            $dataTable = DataTables::eloquent(
                        Terminal::query()->where('air_transportation_id', auth()->user()->id)
                    )
                    ->addIndexColumn()
                    ->editColumn('is_active', function(Terminal $terminal) {
                        if ($terminal->is_active) {
                            return '<a href="javascript:;" class="js-active" data-id="'.$terminal->getKey().'">
                                    <span class="px-4 lg:py-2 py-2 text-sm font-medium leading-5 text-center text-green-700 transition-colors duration-150 bg-white border border-green-700 rounded-md active:bg-green-700 hover:text-white hover:bg-green-700 focus:outline-none focus:shadow-outline-purple">Active</span>
                                </a>';
                        }

                        return '<a href="javascript:;" class="js-active" data-id="'.$terminal->getKey().'">
                                    <span class="px-4 lg:py-2 py-2 text-sm font-medium leading-5 text-center text-red-700 transition-colors duration-150 bg-white border border-red-700 rounded-md active:bg-red-700 hover:text-white hover:bg-red-700 focus:outline-none focus:shadow-outline-purple">Inactive</span>
                                </a>';
                    })
                    ->filterColumn('is_active', function($query, $keyword) {
                        $query->where('is_active', $keyword === "true");
                    })
                    ->filterColumn('updated_at', function($query, $keyword) {
                        $dates = explode('TO', $keyword);
                        if(count($dates) == 2){
                            [$startDate, $endDate] = explode('TO', $keyword);
                        
                            $startDate = Carbon::createFromFormat('Y-m-d', $startDate)->startOfDay();
                            $endDate = Carbon::createFromFormat('Y-m-d', $endDate)->endOfDay();
                            $query->whereBetween('updated_at', [$startDate, $endDate]);
                        }
                    })
                    ->editColumn('updated_at', function (Terminal $terminal) {
                        return [
                            'display' => $terminal->updated_at->format('F j, Y'),
                            'timestamp' => $terminal->updated_at->timestamp
                        ];
                    })
                    ->addColumn('action', function(Terminal $terminal) {
                        return '
                        <div class="flex items-center justify-right ml-auto">
                        
                        <a href="'.route('airtransport.terminals.edit', $terminal->id).'" 
                            class="ml-2 h-8 w-8 text-center flex items-center font-semibold leading-tight rounded-md text-xs text-red-700 bg-red-100 cursor-pointer hover:text-white hover:bg-red-700 transition-all ease-in-out duration-300">
                            <i class="fa-solid fa-pencil m-auto"></i>
                        </a>
                        <a href="javascript:;" data-content-page-id="'.$terminal->id.'"
                        class="ml-2 h-8 w-8 text-center flex items-center font-semibold leading-tight rounded-md text-xs text-orange-700 bg-orange-100 cursor-pointer hover:text-white hover:bg-orange-700 transition-all ease-in-out duration-300">
                        <i class="fa-solid fa-trash mx-auto"></i></a>
                        </div>
    
                        ';
                    })
                    ->rawColumns(['is_active', 'action'])
                    ->make(true);
                    return $dataTable;
        }

        return Inertia::render('airtransport/Location', [
            'data' => $dataTable,
            'activeOptions' =>  [
                [
                    "text"=> "Active",
                    "value"=> true
                ],
                [
                    "text"=> "Inactive",
                    "value"=> false
                ]
            ]

        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return Inertia::render('airtransport/AddLocation', [
            'isCreate'=> true
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(TerminalRequest $request)
    {
        // dd($request);
        $data = Terminal::create([
            'air_transportation_id' => auth()->user()->id,
            'pickup_location' => $request->pickup_location,
            'dropoff_location' => $request->dropoff_location,
            'terminal_type' => $request->terminal_type,
            'is_active' => 1
        ]);

        $images = [];
        if ($request->terminal_images){
            foreach($request->terminal_images as $key => $image)
            {
                $fileName = time()."-".$image->getClientOriginalName();
                $filePath = 'uploads/locations/' . $fileName;
                $path = Storage::disk('s3')->put($filePath, file_get_contents($image));
                $url = Storage::disk('s3')->url($filePath);
                $images[]['name'] = $filePath;
            }
        }
        // dd($images);
        foreach ($images as $key => $image) {
            TerminalImage::create([
                'terminal_id' => $data->id,
                'terminal_image' => $image['name']
            ]);
        }

        sleep(1);
        session()->flash('success', __('airtransport.terminal.addsuccess'));
        return Inertia::location(route('airtransport.terminals.index'));

    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Terminal $terminal)
    {
        $terminal_images = $terminal->terminalImages()->select('id','terminal_image')->get();

        return Inertia::render('airtransport/AddLocation', [
            'terminal' => $terminal,
            'terminal_image' => $terminal_images,
            'isCreate' => false
        ]);

    }

    /**
     * Update the specified resource in storage.
     */
    public function update(TerminalRequest $request, string $id)
    {
        Terminal::where('id', $id)->update([
            'pickup_location' => $request->pickup_location,
            'dropoff_location' => $request->dropoff_location,
            'terminal_type' => $request->terminal_type,
        ]);

        $images = [];
        if ($request->terminal_images){
            foreach($request->terminal_images as $image)
            {
                $img_name = $image->getClientOriginalName();
                if (strpos($img_name, 'preset-file') === false) {
                    $fileName = time()."-".$image->getClientOriginalName();
                    $filePath = 'uploads/locations/' . $fileName;
                    Storage::disk('s3')->put($filePath, file_get_contents($image));
                    $images[] = [
                        'terminal_id' => $id,
                        'terminal_image' => $filePath
                    ];
                }
            }
        }
        if($images){
            TerminalImage::insert($images);
        }
       
        sleep(1);
        session()->flash('success', __('airtransport.terminal.updatesuccess'));
        return Inertia::location(route('airtransport.terminals.index'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        Terminal::destroy($id);
        sleep(1);
        session()->flash('success', __('airtransport.terminal.deletesuccess'));
        return Inertia::location(route('airtransport.terminals.index'));

    }

    public function active(Terminal $terminal)
    {
        $terminal->is_active = !$terminal->is_active;
        $terminal->save();

        return response()->noContent();
    }

    public function deleteImage(Request $request)
    {
        
        $deleteImage = TerminalImage::where('id',$request->img_id)->first();
        Storage::disk('s3')->delete($deleteImage->terminal_image);
        $deleteImage->delete();
    }
}
