<?php

namespace App\Http\Controllers\AirTransport;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Inertia\Inertia;
use App\Http\Requests\AddEditPilotRequest;
use App\Models\Pilot;
use Illuminate\Support\Facades\Storage;
use Yajra\DataTables\Facades\DataTables;

class PilotController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $dataTable = [];
        if ($request->ajax()) {
            $dataTable = DataTables::eloquent(
                Pilot::query()->where('air_transportation_id', auth()->user()->id)
            )
                ->addIndexColumn()
                ->editColumn('is_active', function (Pilot $pilot) {
                    if ($pilot->is_active) {
                        return '<a href="javascript:;" class="js-active" data-id="' . $pilot->getKey() . '">
                                    <span class="px-4 lg:py-2 py-2 text-sm font-medium leading-5 text-center text-green-700 transition-colors duration-150 bg-white border border-green-700 rounded-md active:bg-green-700 hover:text-white hover:bg-green-700 focus:outline-none focus:shadow-outline-purple">Active</span>
                                </a>';
                    }

                    return '<a href="javascript:;" class="js-active" data-id="' . $pilot->getKey() . '">
                                    <span class="px-4 lg:py-2 py-2 text-sm font-medium leading-5 text-center text-red-700 transition-colors duration-150 bg-white border border-red-700 rounded-md active:bg-red-700 hover:text-white hover:bg-red-700 focus:outline-none focus:shadow-outline-purple">Inactive</span>
                                </a>';
                })
                ->filterColumn('is_active', function ($query, $keyword) {
                    $query->where('is_active', $keyword === "true");
                })

                ->addColumn('action', function (Pilot $pilot) {
                    return '
                        <div class="flex items-center justify-right ml-auto">
                        <a href="'.route('airtransport.pilots.show', $pilot).'" 
                                class="ml-2 h-8 w-8 text-center flex items-center font-semibold leading-tight rounded-md text-xs text-orange-700 bg-orange-100 cursor-pointer hover:text-white hover:bg-orange-700 transition-all ease-in-out duration-300">
                                <i class="fa-solid fa-eye mx-auto"></i>
                            </a>

                        <a href="' . route('airtransport.pilots.edit', $pilot->id) . '" 
                            class="ml-2 h-8 w-8 text-center flex items-center font-semibold leading-tight rounded-md text-xs text-red-700 bg-red-100 cursor-pointer hover:text-white hover:bg-red-700 transition-all ease-in-out duration-300">
                            <i class="fa-solid fa-pencil m-auto"></i>
                        </a>
                        <a href="javascript:;" data-content-page-id="' . $pilot->id . '"
                        class="ml-2 h-8 w-8 text-center flex items-center font-semibold leading-tight rounded-md text-xs text-orange-700 bg-orange-100 cursor-pointer hover:text-white hover:bg-orange-700 transition-all ease-in-out duration-300">
                        <i class="fa-solid fa-trash mx-auto"></i></a>
                        </div>
    
                        ';
                })
                ->rawColumns(['is_active', 'action'])
                ->make(true);
            return $dataTable;
        }

        return Inertia::render('airtransport/Pilot', [
            'data' => $dataTable,
            'activeOptions' =>  [
                [
                    "text" => "Active",
                    "value" => true
                ],
                [
                    "text" => "Inactive",
                    "value" => false
                ]
            ]

        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return Inertia::render('airtransport/AddPilot', [
            'isCreate' => true
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(AddEditPilotRequest $request)
    {
        $profilefile = $request->file('profile_image');
        $profilefileName = time() . "-" . $request->profile_image->getClientOriginalName();
        $profilefilePath = 'uploads/pilotimages/' . $profilefileName;
        $profilepath = Storage::disk('s3')->put($profilefilePath, file_get_contents($profilefile));
        $profileurl = Storage::disk('s3')->url($profilefilePath);

        $frontfile = $request->file('license_front_image');
        $frontfileName = time() . "-" . $request->license_front_image->getClientOriginalName();
        $frontfilePath = 'uploads/pilotimages/' . $frontfileName;
        $frontpath = Storage::disk('s3')->put($frontfilePath, file_get_contents($frontfile));
        $fronturl = Storage::disk('s3')->url($frontfilePath);

        $backfile = $request->file('license_back_image');
        $backfileName = time() . "-" . $request->license_back_image->getClientOriginalName();
        $backfilePath = 'uploads/pilotimages/' . $backfileName;
        $backpath = Storage::disk('s3')->put($backfilePath, file_get_contents($backfile));
        $backurl = Storage::disk('s3')->url($backfilePath);

        Pilot::create([
            'air_transportation_id' => auth()->user()->id,
            'pilot_name' => $request->pilot_name,
            'pilot_mobile' => $request->pilot_mobile,
            'nationality' => $request->nationality,
            'experience' => $request->experience,
            'license_front_image' => $frontfilePath,
            'license_back_image' => $backfilePath,
            'profile_image' => $profilefilePath

        ]);
        sleep(1);
        session()->flash('success', __('airtransport.pilots.addsuccess'));
        return Inertia::location(route('airtransport.pilots.index'));
    }

    /**
     * Display the specified resource.
     */
    public function show(Pilot $pilot)
    {
        return Inertia::render('airtransport/ViewPilot', [
            'pilot'=> $pilot,
        ]);

    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Pilot $pilot)
    {
        return Inertia::render('airtransport/AddPilot', [
            'pilot' => $pilot,
            'isCreate' => false
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(AddEditPilotRequest $request, Pilot $pilot)
    {
        $profilefilePath = $pilot->profile_image;
        $frontfilePath = $pilot->license_front_image;
        $backfilePath = $pilot->license_back_image;
        if ($request->hasFile('profile_image')) {
            Storage::disk('s3')->delete($pilot->profile_image);
            $profilefile = $request->file('profile_image');
            $profilefileName = time() . "-" . $request->profile_image->getClientOriginalName();
            $profilefilePath = 'uploads/pilotimages/' . $profilefileName;
            $profilepath = Storage::disk('s3')->put($profilefilePath, file_get_contents($profilefile));
            $profileurl = Storage::disk('s3')->url($profilefilePath);
        }
        if ($request->hasFile('license_front_image')) {
            $img_name = $request->license_front_image->getClientOriginalName();
            if (strpos($img_name, 'preset-file') === false) {
                Storage::disk('s3')->delete($pilot->license_front_image);
                $frontfile = $request->file('license_front_image');
                $frontfileName = time() . "-" . $request->license_front_image->getClientOriginalName();
                $frontfilePath = 'uploads/pilotimages/' . $frontfileName;
                $frontpath = Storage::disk('s3')->put($frontfilePath, file_get_contents($frontfile));
                $fronturl = Storage::disk('s3')->url($frontfilePath);
            }
        }
        if ($request->hasFile('license_back_image')) {
            $img_name = $request->license_front_image->getClientOriginalName();
            if (strpos($img_name, 'preset-file') === false) {
                Storage::disk('s3')->delete($pilot->license_back_image);
                $backfile = $request->file('license_back_image');
                $backfileName = time() . "-" . $request->license_back_image->getClientOriginalName();
                $backfilePath = 'uploads/pilotimages/' . $backfileName;
                $backpath = Storage::disk('s3')->put($backfilePath, file_get_contents($backfile));
                $backurl = Storage::disk('s3')->url($backfilePath);
            }
        }
        $pilot->update([
            'air_transportation_id' => auth()->user()->id,
            'pilot_name' => $request->pilot_name,
            'pilot_mobile' => $request->pilot_mobile,
            'nationality' => $request->nationality,
            'experience' => $request->experience,
            'license_front_image' => $frontfilePath,
            'license_back_image' => $backfilePath,
            'profile_image' => $profilefilePath

        ]);
        sleep(1);
        session()->flash('success', __('airtransport.pilots.updatesuccess'));
        return Inertia::location(route('airtransport.pilots.index'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        Pilot::destroy($id);
        sleep(1);
        session()->flash('success', __('airtransport.pilots.deletesuccess'));
        return Inertia::location(route('airtransport.pilots.index'));
    }

    public function active(Pilot $pilot)
    {
        $pilot->is_active = !$pilot->is_active;
        $pilot->save();

        return response()->noContent();
    }
}
