<?php

namespace App\Http\Controllers\AirTransport;

use App\Http\Controllers\Controller;
use App\Http\Requests\AddEditJetRequest;
use App\Models\Jet;
use App\Models\JetImage;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Inertia\Inertia;
use Yajra\DataTables\Facades\DataTables;

class JetController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $dataTable = [];
        if ($request->ajax()) {
            $dataTable = DataTables::eloquent(
                        Jet::query()->where('air_transportation_id', auth()->user()->id)
                    )
                    ->addIndexColumn()
                    ->editColumn('jet_type', function(Jet $jet) {
                        return getJetTypeText($jet->jet_type);
                    })
                    ->editColumn('routing_type', function(Jet $jet) {
                        return getRoutingTypeText($jet->routing_type);
                    })
                    ->editColumn('is_active', function(Jet $jet) {
                        if ($jet->is_active) {
                            return '<a href="javascript:;" class="js-active" data-id="'.$jet->getKey().'">
                                    <span class="px-4 lg:py-2 py-2 text-sm font-medium leading-5 text-center text-green-700 transition-colors duration-150 bg-white border border-green-700 rounded-md active:bg-green-700 hover:text-white hover:bg-green-700 focus:outline-none focus:shadow-outline-purple">Active</span>
                                </a>';
                        }

                        return '<a href="javascript:;" class="js-active" data-id="'.$jet->getKey().'">
                                    <span class="px-4 lg:py-2 py-2 text-sm font-medium leading-5 text-center text-red-700 transition-colors duration-150 bg-white border border-red-700 rounded-md active:bg-red-700 hover:text-white hover:bg-red-700 focus:outline-none focus:shadow-outline-purple">Inactive</span>
                                </a>';
                    })
                    ->filterColumn('is_active', function($query, $keyword) {
                        $query->where('is_active', $keyword === "true");
                    })
                   
                    ->addColumn('action', function(Jet $jet) {
                        return '
                        <div class="flex items-center justify-right ml-auto">
                        
                        <a href="'.route('airtransport.jets.edit', $jet->id).'" 
                            class="ml-2 h-8 w-8 text-center flex items-center font-semibold leading-tight rounded-md text-xs text-red-700 bg-red-100 cursor-pointer hover:text-white hover:bg-red-700 transition-all ease-in-out duration-300">
                            <i class="fa-solid fa-pencil m-auto"></i>
                        </a>
                        <a href="javascript:;" data-content-page-id="'.$jet->id.'"
                        class="ml-2 h-8 w-8 text-center flex items-center font-semibold leading-tight rounded-md text-xs text-orange-700 bg-orange-100 cursor-pointer hover:text-white hover:bg-orange-700 transition-all ease-in-out duration-300">
                        <i class="fa-solid fa-trash mx-auto"></i></a>
                        </div>
    
                        ';
                    })
                    ->rawColumns(['is_active', 'action', 'jet_type','routing_type'])
                    ->make(true);
                    return $dataTable;
        }

        return Inertia::render('airtransport/Jet', [
            'data' => $dataTable,
            'activeOptions' =>  [
                [
                    "text"=> "Active",
                    "value"=> true
                ],
                [
                    "text"=> "Inactive",
                    "value"=> false
                ]
            ]

        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return Inertia::render('airtransport/AddJet', [
            'isCreate'=> true
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(AddEditJetRequest $request)
    {
        $jet = Jet::create([
            'air_transportation_id' => auth()->user()->id,
            'jet_name' => $request->jet_name,
            'jet_type' => $request->jet_type,
            'routing_type' => $request->routing_type,
            'total_capacity' => $request->total_capacity,
        ]);

        $images = [];
        if ($request->jet_images){
            foreach($request->jet_images as $key => $image)
            {
                $fileName = time()."-".$image->getClientOriginalName();
                $filePath = 'uploads/jets/' . $fileName;
                $path = Storage::disk('s3')->put($filePath, file_get_contents($image));
                $url = Storage::disk('s3')->url($filePath);
                $images[]['jet_image'] = $filePath;
            }
            $jet->jetImages()->createMany($images);
        }
        // dd($images);
       

        sleep(1);
        session()->flash('success', __('airtransport.jets.addsuccess'));
        return Inertia::location(route('airtransport.jets.index'));
    }


    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Jet $jet)
    {
        $jet_images = $jet->jetImages()->select('jet_image')->get();
        return Inertia::render('airtransport/AddJet', [
            'jet' => $jet,
            'jet_image' => $jet_images,
            'isCreate' => false
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(AddEditJetRequest $request, Jet $jet)
    {
        $jet->update([
            'jet_name' => $request->jet_name,
            'jet_type' => $request->jet_type,
            'routing_type' => $request->routing_type,
            'total_capacity' => $request->total_capacity,
        ]);

        $images = [];
        if ($request->jet_images){
            foreach($request->jet_images as $key => $image)
            {
                $img_name = $image->getClientOriginalName();
                if (strpos($img_name, 'preset-file') === false) {
                    $fileName = time()."-".$image->getClientOriginalName();
                    $filePath = 'uploads/jets/' . $fileName;
                    $path = Storage::disk('s3')->put($filePath, file_get_contents($image));
                    $url = Storage::disk('s3')->url($filePath);
                    $images[]['jet_image'] = $filePath;
                }
            }
            $jet->jetImages()->createMany($images);
        }
        // dd($images);
       
        
        sleep(1);
        session()->flash('success', __('airtransport.jets.updatesuccess'));
        return Inertia::location(route('airtransport.jets.index'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        Jet::destroy($id);
        sleep(1);
        session()->flash('success', __('airtransport.jets.deletesuccess'));
        return Inertia::location(route('airtransport.jets.index'));
    }

    public function active(Jet $jet)
    {
        $jet->is_active = !$jet->is_active;
        $jet->save();

        return response()->noContent();
    }

    public function deleteImage(Request $request)
    {
        $jet_images = JetImage::select('id')->where('jet_id',$request->id)->get()->toArray();
        $deleteId=$jet_images[$request->img_id];
        $data = JetImage::where('id',$deleteId)->first();
        Storage::disk('s3')->delete($data->jet_image);
        JetImage::where('id',$deleteId)->delete();
    }
}
