<?php

namespace App\Http\Controllers\AirTransport;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Requests\ChangeQuoteStatusRequest;
use App\Http\Requests\SubmitQuoteRequest;
use App\Models\AirTransportationBooking;
use App\Models\Quote;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;
use Inertia\Inertia;
use Carbon\Carbon;

class HomeController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth:airtransport');
    }

    /**
     * Show the application dashboard.
     *
     * @return \Illuminate\Contracts\Support\Renderable
     */
   

    public function index()
    {
        $userId = auth()->user()->id;
        $bookings = AirTransportationBooking::select([
            'air_transportation_bookings.id',
            'booking_no',
            'user_id',
            'air_transportation_bookings.air_transportation_id',
            'pickup_location',
            'dropoff_location',
            'flying_type',
            'comment',
            'adult',
            'child',
            'infants',
            'trip_type',
            'booking_price',
            'status',
            DB::raw("DATE_FORMAT(departure_date_time, '%e %M %Y, %h:%i %p') as formatted_departure_datetime"),
            DB::raw("DATE_FORMAT(arrival_date_time, '%e %M %Y, %h:%i %p') as formatted_arrival_datetime"),
            DB::raw("DATE_FORMAT(air_transportation_bookings.created_at, '%e %M %Y') as created_datetime"),
            'air_transport_booking_requests.id as booking_request_id',
            'air_transport_booking_requests.air_transportation_id as air_transportation_id_request',
            'quotes.id as quote_id',
            'quotes.company_quote_price',
            'quotes.quote_status as quote_status',
            'quotes.user_quote_price',
            'quotes.quote_date_time',

        ])
        ->leftJoin('air_transport_booking_requests', function ($join) use($userId){
            $join->on('air_transport_booking_requests.air_booking_id', '=', 'air_transportation_bookings.id')
                ->where('air_transport_booking_requests.air_transportation_id', '=', $userId);
        })
        ->leftJoin('quotes', function ($join) use($userId){
            $join->on('quotes.air_transportation_booking_id', '=', 'air_transportation_bookings.id')
                ->where('quotes.air_transportation_id', '=', $userId)
                ->whereNotIn('quotes.quote_status', [ Quote::REJECTED, Quote::AUTO_REJECTED]);
        })
        ->whereNotNull('air_transport_booking_requests.id')
        ->with([
            'users:id,name,sendbird_chat_id',
        ])
        ->where(function ($query) use ($userId) {
            $query->whereNull('air_transportation_bookings.air_transportation_id')
                ->orWhere('air_transportation_bookings.air_transportation_id', $userId);
        })
        ->whereNotIn('air_transportation_bookings.status', [3,4])
        ->orderBy('id', 'desc');
      
        return Inertia::render('airtransport/AirDashboard', [
            'bookings' => $bookings->get(),
            'upcoming_bookings' => $bookings->where('status', 2)->where('air_transportation_bookings.air_transportation_id', $userId)->get(),
            'currUserId' => auth()->user()->sendbird_chat_id
        ]);
    }


    public function submitQuote(SubmitQuoteRequest $request)
    {
        $booking = AirTransportationBooking::findOrFail($request->booking_id);
        $quoteStatus = config('constant.quote_status');
        $res = Quote::create([
            'air_transportation_booking_id' => $request->booking_id,
            'air_transportation_id' => auth()->user()->id,
            'company_quote_price' => $request->quote,
            'quote_date_time' =>  Carbon::now()->format('Y-m-d H:i:s'),
            'quote_status' => $quoteStatus['counter'] //quote status for counter
        ]);
        sendPushNotification(auth()->user()->id, $booking->user_id, 'Quote received', 'company given a quotation', $res->id, 'quote', 'airtransport', 'customer', 3);
        session()->flash('success', __('airtransport.quote.success'));
        return Inertia::location(route('airtransport.home'));
    }

    public function changeStatus(ChangeQuoteStatusRequest $request)
    {
        try {
            DB::beginTransaction();
            $bookingStatus = config('constant.booking_status');
            $quotes = Quote::where('id', $request->id)->first();
            $booking = AirTransportationBooking::findOrFail($quotes->air_transportation_booking_id);
            if($booking->status != $bookingStatus['pending']){
                DB::rollBack()  ;
                return response()->json([
                    'success' => false,
                    'message' => __('airtransport.quote.already_processed'),
                    'data' => $quotes->air_transportation_booking_id
                ]);
            }

            $quotes->update(['quote_status' => $request->status]);

            if ($request->status == Quote::ACCEPTED) {

                $quotes->where('air_transportation_booking_id', $quotes->air_transportation_booking_id)
                ->whereNotIN('quote_status',[Quote::ACCEPTED, Quote::REJECTED] )
                ->update(['quote_status' => Quote::AUTO_REJECTED]);

                $booking->update([
                    'status' => $bookingStatus['upcoming'], //booking status for upcoming
                    'air_transportation_id' => auth()->user()->id, //booking status for upcoming
                    'booking_price' => $quotes->user_quote_price
                ]);

                sendPushNotification(auth()->user()->id, $booking->user_id, 'Quote accepted', 'company accepted quote', $quotes->air_transportation_booking_id, 'airbooking', 'airtransport', 'customer', 3);
                $message = __('airtransport.quote.accepted');
            }
            else{
                sendPushNotification(auth()->user()->id, $booking->user_id, 'Quote rejected', 'company rejected quote', $request->id, 'quote', 'airtransport', 'customer', 3);
                $message = __('airtransport.quote.rejected');

            }
            DB::commit();

            return response()->json([
                'success' => true,
                'message' => $message,
                'data' => $quotes->air_transportation_booking_id
            ]);
        } catch (\Exception $e) {
            // Log the error if needed
            // Log::error($e->getMessage());
            DB::rollBack()  ;
            return response()->json([
                'message' => $e->getMessage(),
            ], 500);
        }
    }
}
