<?php

namespace App\Http\Controllers\AirTransport;

use App\Http\Controllers\Controller;
use App\Http\Requests\CancelTripRequest;
use App\Http\Requests\ChangeQuoteStatusRequest;
use App\Models\AirTransportationBooking;
use App\Models\CancelBooking;
use App\Models\CancelReason;
use App\Models\Quote;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;

class BookingController extends Controller
{
    public function getBookings()
    {
        $type = request()->input('type');
        $userId = auth()->user()->id;

        $bookings = AirTransportationBooking::with([
            'users:id,name',
        ])
        ->select([
            'air_transportation_bookings.id',
            'booking_no',
            'user_id',
            'air_transportation_bookings.air_transportation_id',
            'pickup_location',
            'dropoff_location',
            'flying_type',
            'comment',
            'adult',
            'child',
            'infants',
            'trip_type',
            'booking_price',
            'status',
            DB::raw("DATE_FORMAT(departure_date_time, '%e %M %Y, %h:%i %p') as formatted_departure_datetime"),
            DB::raw("DATE_FORMAT(arrival_date_time, '%e %M %Y, %h:%i %p') as formatted_arrival_datetime"),
            DB::raw("DATE_FORMAT(air_transportation_bookings.created_at, '%e %M %Y') as created_datetime"),
            'air_transport_booking_requests.id as booking_request_id',
            'air_transport_booking_requests.air_transportation_id as air_transportation_id_request',
            'quotes.id as quote_id',
            'quotes.company_quote_price',
            'quotes.quote_status as quote_status',
            'quotes.user_quote_price',
            'quotes.quote_date_time',
        ])
        ->rightJoin('air_transport_booking_requests', function ($join) use($userId){
            $join->on('air_transport_booking_requests.air_booking_id', '=', 'air_transportation_bookings.id')
                 ->where('air_transport_booking_requests.air_transportation_id', '=', $userId);
        })
        ->rightJoin('quotes', function ($join) use($userId){
            $join->on('quotes.air_transportation_booking_id', '=', 'air_transportation_bookings.id')
                 ->where('quotes.air_transportation_id', '=', $userId);
        })
        ->when($type == null || $type == 'pending', function ($q) use ($userId) {
            $q->where('status', 2)
                ->where('air_transportation_bookings.air_transportation_id', $userId)
                ->where('departure_date_time', '>', date('Y-m-d H:i:s'))
                ->whereIn('quotes.quote_status', [ Quote::ACCEPTED]);
        })
        ->when($type != null && $type == 'ongoing', function ($q) use ($userId) {
            $q->where('status', 2)
                ->where('air_transportation_bookings.air_transportation_id', $userId)
                ->where('departure_date_time', '<=', date('Y-m-d H:i:s'))
                ->whereIn('quotes.quote_status', [Quote::ACCEPTED]);
        })
        ->when($type != null && $type == 'past', function ($q){
            $q->where(function($q){
                $q->whereIn('status', [3,4])
                ->orWhereIn('quotes.quote_status', [ Quote::REJECTED, Quote::AUTO_REJECTED]);
            });
        })
        ->whereNotNull('air_transportation_bookings.id')
        ->orderBy('air_transportation_bookings.id', 'desc')
        ->paginate(config('constant.PAGINATION_LIMIT'));
        
        if ($type != null) {
            return response()->json([
                'success' => false,
                'message' => __('airtransport.quote.already_processed'),
                'bookings' => $bookings
            ]);
        }

        $cancel_reasons = CancelReason::where('type', 3)->get();
        return Inertia::render('airtransport/AirBooking', [
            'bookings' => $bookings,
            'cancel_reason' => $cancel_reasons
        ]);
    }

    public function cancelTrip(CancelTripRequest $request)
    {
        try {
            $booking = AirTransportationBooking::findOrFail($request->booking_id);
            $cancelledBy = config('constant.cancelled_by');
            $cancelType = config('constant.cancel_type');
            $bookingStatus = config('constant.booking_status');
           
            $booking->cancel_booking()->create([
                'cancel_reason_id' => $request->reason,
                'cancel_description' => $request->description,
                'cancelled_by' => $cancelledBy['airtransport'], 
                'type' => $cancelType['airtransport'],  
            ]);
            $booking->cancel_reason = $request->description;
            $booking->status = $bookingStatus['cancelled']; //booking status for cancel trip
            $booking->save();
            sendPushNotification(auth()->user()->id, $booking->user_id, 'Booking Cancelled', 'company cancelled your booking', $booking->id, 'airbooking', 'airtransport', 'customer', 3);
            session()->flash('message', __('airtransport.booking.cancel'));

            return redirect()->route('airtransport.bookings');
        } catch (\Exception $e) {

            session()->flash('message', __('airtransport.booking.error'));
            return redirect()->route('airtransport.bookings');
        }
    }

    public function changeStatus(ChangeQuoteStatusRequest $request)
    {
        try {
            $booking = AirTransportationBooking::where('id', $request->id)->first();
            $booking->update(['status' => $request->status]);
            sendPushNotification(auth()->user()->id, $booking->user_id, 'Booking Completed', 'company completed your booking', $booking->id, 'airbooking', 'airtransport', 'customer', 3);
            return response()->json([
                'message' => "__('airtransport.booking.complete')",
                'data' => $booking->id
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => "__('airtransport.booking.error')",
            ], 500);
        }
    }
}
