<?php

namespace App\Http\Controllers\AirTransport\Auth;

use App\Models\AirTransportation;
use App\Http\Controllers\Controller;
use App\Models\Day;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use App\Providers\RouteServiceProvider;
use App\Services\SendbirdService;
use App\Services\StripeService;
use Exception;
use Illuminate\Auth\Events\Registered;
use Illuminate\Support\Facades\Validator;
use Illuminate\Foundation\Auth\RegistersUsers;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Inertia\Inertia;
use SpaceO\RESTAuth\Traits\SendOTPTrait;
use Twilio\Rest\Client;

class RegisterController extends Controller
{
    use SendOTPTrait;

    private $stripeService;
    private $sendbirdService;
    private int $otpLength;

    /*
    |--------------------------------------------------------------------------
    | Register Controller
    |--------------------------------------------------------------------------
    |
    | This controller handles the registration of new users as well as their
    | validation and creation. By default this controller uses a trait to
    | provide this functionality without requiring any additional code.
    |
    */

    use RegistersUsers;

    /**
     * Where to redirect users after registration.
     *
     * @var string
     */
    protected $redirectTo = RouteServiceProvider::OTP;

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct(StripeService $stripeService, SendbirdService $sendbirdService)
    {
        $this->middleware('guest:airtransport');
        $this->stripeService = $stripeService;
        $this->sendbirdService = $sendbirdService;
        $this->otpLength = config('rest_auth.air_otp_length');
    }

    public function showRegistrationForm()
    {
        return Inertia::render('airtransport/Signup', [
            'step' => session()->get('step'),
            'type' => session()->get('type'),
            'user_id' => session()->get('user_id'),
            'user_mobile' => session()->get('user_mobile')
        ]);
    }
    /**
     * Handle a registration request for the application.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Http\JsonResponse
     */
    public function register(Request $request)
    {
        $this->validator($request->all())->validate();

        event(new Registered($user = $this->create($request->all())));
        $otp = $this->sendOTPOnMobile($request);
        AirTransportation::where('id', $user['id'])->update(
            array_combine($this->OTPFields(), [
                $otp, now()->addMinutes(config('rest_auth.otp_expires_in'))
            ])
        );
        $days = ['sunday','monday','tuesday','wednesday','thursday','friday','saturday'];
        $data = array_map(function ($day) use ($user) {
            return [
                'air_transportation_id' => $user->id,
                'days' => $day,
                'created_at' => now(),
                'updated_at' => now()
            ];
        }, $days);
        Day::insert($data);
        session()->put('user_id', $user->id);
        session()->put('user_mobile', $user->mobile);
        session()->put('isd_code', $user->isd_code);
        $this->guard()->login($user);
       
        if ($response = $this->registered($request, $user)) {
            return $response;
        }

        return $request->wantsJson()
            ? new JsonResponse([], 201)
            : redirect($this->redirectPath());
    }


    /**
     * Get a validator for an incoming registration request.
     *
     * @param  array  $data
     * @return \Illuminate\Contracts\Validation\Validator
     */
    protected function validator(array $data)
    {
        return Validator::make($data, [
            'type' => ['required', 'in:1,2'],
            'first_name' => ['required', 'string', 'max:255'],
            'last_name' => ['required', 'string', 'max:255'],
            'mobile' => ['required' , 'unique:air_transportations'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:air_transportations'],
            'password' => ['required', 'string', 'min:8', 'confirmed'],
        ]);
    }

    /**
     * Create a new user instance after a valid registration.
     *
     * @param  array  $data
     * @return \App\Models\User
     */
    protected function create(array $data)
    {
        $stripeCustomer = $this->stripeService->createCustomer($data['email'], $data['first_name'], 1);
        $profilefilePath = "";
        if ($data['profile_image'] != "") {
            $profilefile = $data['profile_image'];
            $profilefileName = time() . "-" . $data['profile_image']->getClientOriginalName();
            $profilefilePath = 'uploads/profileimages/' . $profilefileName;
            Storage::disk('s3')->put($profilefilePath, file_get_contents($profilefile));
        }

        $profilePhoto = $data['profile_image'] ? config('services.aws.aws_cloud_url') .$profilefilePath : "https://sendbird.com/main/img/profiles/profile_05_512px.png";
        $sendbirdUser = $this->sendbirdService->createUser($data['first_name'], $data['first_name'], $profilePhoto);

        return AirTransportation::create([
            'uuid' => GetUuid(),
            'first_name' => $data['first_name'],
            'last_name' => $data['last_name'],
            'email' => $data['email'],
            'mobile' => $data['mobile'],
            'isd_code' => $data['isd_code'],
            'password' => Hash::make($data['password']),
            'step' => 1,
            'type' => $data['type'],
            'sendbird_chat_id' => $sendbirdUser['user_id'],
            'stripe_id' => $stripeCustomer->id,
            'profile_image' => $profilefilePath
        ]);
       
        
    }

    protected function guard()
    {
        return Auth::guard('airtransport');
    }

    public function sendOTPOnMobile($request)
    {
        $otp = $this->generateOTP();
        
        // Send SMS on mobile number via HTTP client
        $receiverNumber = '"' . $request->isd_code . $request->mobile . '"';
        // dd($receiverNumber);
        $message = __('airtransport.otp.otp_message') . ' ' . $otp;
        try {

            $account_sid = config('services.twilio.account_sid');
            $auth_token = config('services.twilio.auth_token');
            $twilio_number = config('services.twilio.twilio_number');

            $client = new Client($account_sid, $auth_token);
            $client->messages->create($receiverNumber, [
                'from' => $twilio_number,
                'body' => $message
            ]);

            // dd('SMS Sent Successfully.');

        } catch (Exception $e) {
            // dd("Error: ". $e->getMessage());
        }
        return $otp;
    }

    
}
