<?php

namespace App\Http\Controllers\Admin;

use Carbon\Carbon;
use App\Models\User;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Yajra\DataTables\Facades\DataTables;
use Inertia\Inertia;

class UserController extends Controller
{

    public function __construct()
    {
        $this->middleware(['permission:User List|User View|User Update'],['only' => ['index']]);     
        $this->middleware(['permission:User View'],['only' => ['show']]);     
        $this->middleware(['permission:User Update'],['only' => ['active']]);     
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {

        if ($request->ajax()) {
            return DataTables::eloquent(
                User::query()
            )
                ->addIndexColumn()
                ->addColumn('total_rides', function (User $user) {
                    $totalRides = $user->bookings->count();

                    return $totalRides;
                })
                ->editColumn('is_active', function (User $user) {
                    if ($user->is_active) {
                        return '<a href="javascript:;" class="js-active" data-id="' . $user->getKey() . '">
                        <span class="px-4 lg:py-2 py-2 text-sm font-medium leading-5 text-center text-green-700 transition-colors duration-150 bg-white border border-green-700 rounded-md active:bg-green-700 hover:text-white hover:bg-green-700 focus:outline-none focus:shadow-outline-purple">Active</span>
                    </a>';
                    }

                    return '<a href="javascript:;" class="js-active" data-id="' . $user->getKey() . '">
                    <span class="px-4 lg:py-2 py-2 text-sm font-medium leading-5 text-center text-red-700 transition-colors duration-150 bg-white border border-red-700 rounded-md active:bg-red-700 hover:text-white hover:bg-red-700 focus:outline-none focus:shadow-outline-purple">Inactive</span>
                </a>';
                })
                ->filterColumn('is_active', function ($query, $keyword) {
                    $query->where('is_active', $keyword === "true");
                })
                ->filterColumn('created_at', function ($query, $keyword) {

                    [$startDate, $endDate] = explode('TO', $keyword);

                    $startDate = Carbon::createFromFormat('Y-m-d', $startDate)->startOfDay();
                    $endDate = Carbon::createFromFormat('Y-m-d', $endDate)->endOfDay();
                    $query->whereBetween('created_at', [$startDate, $endDate]);
                })
                ->filterColumn('updated_at', function ($query, $keyword) {
                    $query->whereDate('updated_at', $keyword);
                })
                ->editColumn('created_at', function ($user) {
                    return [
                        'display' => $user->created_at->format('F j, Y'),
                        'timestamp' => $user->created_at->timestamp
                    ];
                })
                ->editColumn('updated_at', function ($user) {
                    return [
                        'display' => $user->updated_at->format('F j, Y'),
                        'timestamp' => $user->updated_at->timestamp
                    ];
                })
                ->addColumn('action', function (User $user) {
                    return '
                        <div class="flex items-center ml-auto">
                            <a href="' . route('admin.users.show', $user) . '" 
                                class="ml-2 h-8 w-8 text-center flex items-center font-semibold leading-tight rounded-md text-xs text-orange-700 bg-orange-100 cursor-pointer hover:text-white hover:bg-orange-700 transition-all ease-in-out duration-300">
                                <i class="fa-solid fa-eye mx-auto"></i>
                            </a>
                        </div>
                        ';
                })
                ->rawColumns(['is_active', 'action'])
                ->make(true);
        }

        return Inertia::render('User', [
            'activeOptions' =>  [
                [
                    "text" => "Active",
                    "value" => true
                ],
                [
                    "text" => "Inactive",
                    "value" => false
                ]
            ]
        ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\User  $user
     * @return \Illuminate\Http\Response
     */
    public function show(User $user)
    {
        $tokens = $user->tokens()->get();

        return Inertia::render('ViewUser', [
            'user' => $user,
            'tokens' => $tokens
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\User  $user
     * @return \Illuminate\Http\Response
     */
    public function edit(User $user)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\User  $user
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, User $user)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\User  $user
     * @return \Illuminate\Http\Response
     */
    public function destroy(User $user)
    {
        //
    }

    public function active(User $user)
    {
        $user->is_active = !$user->is_active;
        $user->save();

        return response()->noContent();
    }
}
