<?php

namespace App\Http\Controllers\Admin;

use App\Models\Admin;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Spatie\Permission\Models\Role;
use App\Http\Controllers\Controller;
use App\Http\Requests\AddEditSubAdminRequest;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Validation\Rules\Unique;
use Yajra\DataTables\Facades\DataTables;
use Inertia\Inertia;

class SubAdminController extends Controller
{

    public function __construct()
    {
        $this->middleware(['permission:Sub Admins List|Sub Admins Update|Sub Admins Delete|Sub Admins Add'],['only' => ['index']]);     
        $this->middleware(['permission:Sub Admins Update'],['only' => ['edit','update','active']]);     
        $this->middleware(['permission:Sub Admins Add'],['only' => ['create','store']]);     
        $this->middleware(['permission:Sub Admins Delete'],['only' => ['destroy']]);     
    }
    
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $dataTable = [];
        if ($request->ajax()) {
            $dataTable = DataTables::eloquent(
                Admin::query()
                    ->with('roles')
                    ->whereDoesntHave('roles', function ($query) {
                        $query->whereIn('name', ['super admin', 'developer']);
                    })
                    ->select('admins.*')
            )
                ->addIndexColumn()
                ->addColumn('roles', function (Admin $admin) {
                    return $admin->roles->map(function ($role) {
                        return '- ' . Str::limit($role->name, 30);
                    })->implode('<br>');
                })
                ->editColumn('is_active', function (Admin $admin) {
                    if ($admin->is_active) {
                        return '<a href="javascript:;" class="js-active" data-id="' . $admin->getKey() . '">
                            <span class="px-4 lg:py-2 py-2 text-sm font-medium leading-5 text-center text-green-700 transition-colors duration-150 bg-white border border-green-700 rounded-md active:bg-green-700 hover:text-white hover:bg-green-700 focus:outline-none focus:shadow-outline-purple">Active</span>
                        </a>';
                    }

                    return '<a href="javascript:;" class="js-active" data-id="' . $admin->getKey() . '">
                        <span class="px-4 lg:py-2 py-2 text-sm font-medium leading-5 text-center text-red-700 transition-colors duration-150 bg-white border border-red-700 rounded-md active:bg-red-700 hover:text-white hover:bg-red-700 focus:outline-none focus:shadow-outline-purple">Inactive</span>
                    </a>';
                })
                ->editColumn('created_at', function ($admin) {
                    return [
                        'display' => $admin->created_at->format('F j, Y'),
                        'timestamp' => $admin->created_at->timestamp
                    ];
                })
                ->editColumn('updated_at', function (Admin $admin) {
                    return [
                        'display' => $admin->updated_at->format('F j, Y'),
                        'timestamp' => $admin->updated_at->timestamp
                    ];
                })
                ->addColumn('action', function (Admin $admin) {
                    return '
                        <div class="flex items-center justify-center ml-auto">
                        <a class="ml-2 h-8 w-8 text-center flex items-center font-semibold leading-tight rounded-md text-xs text-green-700 bg-green-100 cursor-pointer hover:text-white hover:bg-green-700 transition-all ease-in-out duration-300" 
                        href="' . route('admin.sub-admins.edit', $admin) . '" >
                            <i class="fas fa-pencil mx-auto"></i>
                        </a>
                        <a href="javascript:;" data-admin-id="' . $admin->id . '"
                        class="ml-2 h-8 w-8 text-center flex items-center font-semibold leading-tight rounded-md text-xs text-red-700 bg-red-100 cursor-pointer hover:text-white hover:bg-red-700 transition-all ease-in-out duration-300">
                        <i class="fa-solid fa-trash mx-auto"></i></a>
                 </div> 
                        ';
                })
                ->rawColumns(['roles', 'is_active', 'action'])
                ->make(true);
            return $dataTable;
        }
        return Inertia::render('SubAdmin', [
            'admin_data' => $dataTable
        ]);
        // return view('admin.sub_admins.index');
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return Inertia::render('AddSubadmin', [
            'isCreate' => true,
            'roles' => Role::whereNotIn('name', ['super admin', 'developer'])->pluck("name"),
        ]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(AddEditSubAdminRequest $request)
    {
        // dd($request->all());
        try {
            DB::beginTransaction();
            $password = Str::random(6);
            $admin = Admin::create([
                'name' => $request->name,
                'email' => $request->email,
                'password' => bcrypt($password),
                'is_active' => 1,
            ]);

            $admin->assignRole($request->roles);
            $mailData = [
                'email' => $request->email,
                'password' => $password,
                'subject' => 'Sub Admin Account Created for 4U',
            ];

            try {
                Mail::send('email.sub-admin-account', $mailData, function ($message) use ($mailData,$request) {
                    $message->to($request->email)->subject($mailData['subject']);
                });
            } catch (\Exception $e) {
                Log::error($e->getMessage());
            }
            DB::commit();
            session()->flash('success', 'Subadmin added successfully');
            return Inertia::location(route('admin.sub-admins.index'));
        } catch (\Throwable $th) {
            DB::rollBack();
            session()->flash('error', 'Something went wrong, please try again');
            return Inertia::location(route('admin.sub-admins.index'));
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show(Admin $sub_admin)
    {
        return view('admin.sub_admins.show', ['admin' => $sub_admin]);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit(Admin $sub_admin)
    {
        return Inertia::render('AddSubadmin', [
            'isCreate' => false,
            'roles' => Role::whereNotIn('name', ['super admin', 'developer'])->pluck('name'),
            'selected_roles' => $sub_admin->getRoleNames(),
            'subadmin' => $sub_admin
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Admin $sub_admin, AddEditSubAdminRequest $request)
    {
        try {
            DB::beginTransaction();

            $sub_admin->update([
                'name' => $request->name,
                'email' => $request->email,
                'is_active' => $request->is_active,
            ]);

            $sub_admin->syncRoles($request->roles);
            DB::commit();
            session()->flash('success', 'Subadmin updated successfully');
            return Inertia::location(route('admin.sub-admins.index'));
        } catch (\Throwable $th) {
            DB::rollBack();
            session()->flash('error', 'Something went wrong, please try again');
            return Inertia::location(route('admin.sub-admins.index'));
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        Admin::destroy($id);
        session()->flash('success', 'Subadmin deleted successfully');
        return Inertia::location(route('admin.sub-admins.index'));
    }

    public function active(Admin $sub_admin)
    {
        $sub_admin->is_active = !$sub_admin->is_active;
        $sub_admin->save();

        return response()->noContent();
    }
}
