<?php

namespace App\Http\Controllers\Admin;

use App\Models\Setting;
use Illuminate\Http\Request;
use App\Models\AppVersionLog;
use App\Http\Controllers\Controller;
use Illuminate\Database\Eloquent\Casts\Json;
use Illuminate\Support\Facades\Artisan;
use Inertia\Inertia;

class SettingController extends Controller
{


    public function __construct()
    {
        $this->middleware(['permission:Settings Update|Settings List'],['only' => ['index']]); 
        $this->middleware(['permission:Settings Update'],['only' => ['updateCancellation','updateRadius','updateWaiting']]); 
            
    }
    public function index()
    {
        $data = Setting::all();
        $result = [];
        $data->groupBy('key')->map(function ($setting, $key) use (&$result) {
            $values = $setting->value('values');
            if (Setting::APP_VERSIONS == $key) {
                $a = collect($values)->map(function ($appVersions, $appKey) use (&$result) {
                    if (Setting::ANDROID == $appVersions['platform']) {
                        $result['androidVersion'] = $appVersions['version'];
                    } elseif (Setting::IOS ==  $appVersions['platform']) {
                        $result['iosVersion'] = $appVersions['version'];
                    }
                    return $result;
                });
            } elseif (Setting::RADIUS == $key) {
                $result['radius'] = $values['radius'];
            } elseif (Setting::WAITING == $key) {
                $result['waiting_time'] = $values['waiting_time'];
                $result['waiting_charge'] = $values['waiting_charge'];
            } elseif (Setting::CANCELLATION == $key) {
                $result['cancellation_time'] = $values['cancellation_time'];
                $result['cancellation_charge'] = $values['cancellation_charge'];
            }
        });
        return Inertia::render('Setting', $result);
    }

    public function mobileDetails()
    {

        $versions = collect(Setting::where('key', Setting::APP_VERSIONS)->value('values'));
        return view('admin.settings.mobile', [
            'iosVersion' => $versions->where('platform', Setting::IOS)->first(),
            'androidVersion' => $versions->where('platform', Setting::ANDROID)->first(),
        ]);
    }

    public function updateMobileSettings(Request $request)
    {
        $validated = $request->validate([
            'android' => 'required|array',
            'android.version' => [
                'required',
                'integer',
                'min:0',
                function ($attribute, $value, $fail) {
                    if (!Setting::whereVersionShouldBeGreater('android', $value)->count()) {
                        $fail(__('basecode/admin.responses.invalid_app_version_message'));
                    }
                }
            ],
            'android.force_updateable' => 'nullable|in:on',
            'ios' => 'required|array',
            'ios.version' => [
                'required',
                'integer',
                'min:0',
                function ($attribute, $value, $fail) {
                    if (!Setting::whereVersionShouldBeGreater('ios', $value)->count()) {
                        $fail(__('basecode/admin.responses.invalid_app_version_message'));
                    }
                }
            ],
            'ios.force_updateable' => 'nullable|in:on',
        ]);

        Setting::where('key', Setting::APP_VERSIONS)->update([
            'values' => [
                [
                    'platform' => Setting::ANDROID,
                    'version' => (int) $validated['android']['version'],
                    'force_updateable' => isset($validated['android']['force_updateable']) ? true : false,
                ],
                [
                    'platform' => Setting::IOS,
                    'version' => (int) $validated['ios']['version'],
                    'force_updateable' => isset($validated['ios']['force_updateable']) ? true : false,
                ]
            ]
        ]);

        if ($validated['android']['version'] > 0 || $validated['ios']['version'] > 0) {

            AppVersionLog::updateOrCreate([
                'android_version' => (int) $validated['android']['version'],
                'ios_version' => (int) $validated['ios']['version']
            ], [
                'is_android_force_update' => isset($validated['android']['force_updateable']) ? true : false,
                'is_ios_force_update' => isset($validated['ios']['force_updateable']) ? true : false,
            ]);
        }

        return redirect()->back()->withMessage(__('basecode/admin.messages.app_versions_updated'));
    }

    public function updateRadius(Request $request)
    {
        $setting = Setting::where('key', Setting::RADIUS)->first();
        if ($setting) {
            $setting->values = [
                'radius' => $request->radius,
            ];
            $setting->save();
        }
        session()->flash('success', __('basecode/admin.messages.radius_updated'));

        return Inertia::location(route('admin.settings'));
    }

    public function updateWaiting(Request $request)
    {
        $setting = Setting::where('key', Setting::WAITING)->first();
        if ($setting) {
            $setting->values = [
                'waiting_time' => $request->waiting_time,
                'waiting_charge' => $request->waiting_charge,
            ];
            $setting->save();
        }
        session()->flash('success', __('basecode/admin.messages.waiting_updated'));

        return Inertia::location(route('admin.settings'));
    }

    public function updateCancellation(Request $request)
    {
        $setting = Setting::where('key', Setting::CANCELLATION)->first();
        if ($setting) {
            $setting->values = [
                'cancellation_time' => $request->cancellation_time,
                'cancellation_charge' => $request->cancellation_charge,
            ];
            $setting->save();
        }
        session()->flash('success', __('basecode/admin.messages.cancel_updated'));

        return Inertia::location(route('admin.settings'));
    }

    public function runSeeder()
    {
        // Run the specific seeder (change 'YourSeeder' to your seeder class)
        Artisan::call('db:seed', ['--class' => 'AdminSettingSeeder']);

        return 'Seeder executed successfully!';
    }
}
