<?php

namespace App\Http\Controllers\Admin;

use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Spatie\Permission\Models\Role;
use App\Http\Controllers\Controller;
use App\Http\Requests\AddEditRoleRequest;
use Illuminate\Support\Facades\DB;
use Spatie\Permission\Models\Permission;
use Yajra\DataTables\Facades\DataTables;
use Inertia\Inertia;

class RoleController extends Controller
{

    public function __construct()
    {
        $this->middleware(['permission:Roles List|Roles Update|Roles Delete|Roles Add'],['only' => ['index']]);     
        $this->middleware(['permission:Roles Update'],['only' => ['edit','update','active']]);     
        $this->middleware(['permission:Roles Add'],['only' => ['create','store']]);     
        $this->middleware(['permission:Roles Delete'],['only' => ['destroy']]);     
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $permissions = Permission::select('id','name as permission_name')->get();
        $dataTable = [];
        if ($request->ajax()) {
            $dataTable = DataTables::eloquent(
                        Role::whereNotIn('name', ['super admin','developer'])
                    )
                    ->addIndexColumn()
                    ->editColumn('created_at', function (Role $role) {
                        return [
                            'display' => $role->created_at->format('F j, Y'),
                            'timestamp' => $role->created_at->timestamp
                        ];
                    })
                    ->editColumn('updated_at', function (Role $role) {
                        return [
                            'display' => $role->updated_at->format('F j, Y'),
                            'timestamp' => $role->updated_at->timestamp
                        ];
                    })
                    ->addColumn('action', function(Role $role) {
                        return '
                        <div class="flex items-center ml-auto">
                            <a href="'.route('admin.roles.edit', $role->id).'" 
                                class="h-8 w-8 text-center flex items-center font-semibold leading-tight rounded-md text-xs text-red-700 bg-orange-100 cursor-pointer hover:text-white hover:bg-orange-700 transition-all ease-in-out duration-300">
                                <i class="fa-solid fa-pencil m-auto"></i>
                            </a>
                            <a href="javascript:;" data-rolw-id="'.$role->id.'"
                            class="ml-2 h-8 w-8 text-center flex items-center font-semibold leading-tight rounded-md text-xs text-red-700 bg-red-100 cursor-pointer hover:text-white hover:bg-red-700 transition-all ease-in-out duration-300">
                            <i class="fa-solid fa-trash mx-auto"></i></a>
                        </div>
                        ';
                    })
                    ->rawColumns(['action'])
                    ->make(true);
                    return $dataTable;
        }

        // return view('admin.roles.index');
        return Inertia::render('Role', [
            'role_data' => $dataTable,
         ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return Inertia::render('AddRole', [
            'isCreate' => true,
            'permissions' => Permission::pluck('name')
        ]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(AddEditRoleRequest $request)
    {
        try {
            DB::beginTransaction();
            $permissions = $request->permissions;
            $role = Role::create([
                'name' => strtolower($request->name)
            ]);

            $role->syncPermissions($permissions);
            DB::commit();
            session()->flash('success', 'Role updated successfully');
            return Inertia::location(route('admin.roles.index'));
        } catch (\Throwable $th) {
           
            DB::rollBack();
            session()->flash('error', 'Something went wrong, please try again');
            return Inertia::location(route('admin.roles.index'));
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  Spatie\Permission\Models\Role  $role
     * @return \Illuminate\Http\Response
     */
    public function show(Role $role)
    {
        return view('admin.roles.show', [
            'role' => $role
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  Spatie\Permission\Models\Role  $role
     * @return \Illuminate\Http\Response
     */
    public function edit(Role $role)
    {
        return Inertia::render('AddRole', [
            'isCreate' => false,
            'role' => $role->load('permissions'),
            'permissions' => Permission::pluck('name'),
            'selected_permissions' => $role->getPermissionNames(),
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  Spatie\Permission\Models\Role  $role
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Role $role)
    {
        $request->validate([
            'name' => [
                'required',
                'min:4',
                Rule::unique('roles', 'name')->ignore($role->getKey()),
            ],
            'permissions' => [
                'required',
                'array',
                'min:1'
            ],
            'permissions.*' => [
                'required'
            ]
        ]);
        $permissions = $request->permissions;
        $role->update([
            'name' => strtolower($request->name)
        ]);

        $role->syncPermissions($permissions);

        return Inertia::location(route('admin.roles.index'));
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        Role::destroy($id);
        sleep(1);
        session()->flash('success', 'Role has been deleted successfully.');
        return Inertia::location(route('admin.roles.index'));
    }

    public function active(Role $role)
    {
        $role->is_active = !$role->is_active;
        $role->save();

        return response()->noContent();
    }
}
