<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\AddEditPromocodeRequest;
use App\Models\Promocode;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Yajra\DataTables\Facades\DataTables;

class PromocodeController extends Controller
{
    public function __construct()
    {
        $this->middleware(['permission:PromoCodes List|PromoCodes Update|PromoCodes Delete|PromoCodes Add'],['only' => ['index']]);     
        $this->middleware(['permission:PromoCodes Update'],['only' => ['edit','update','active']]);     
        $this->middleware(['permission:PromoCodes Add'],['only' => ['create','store']]);     
        $this->middleware(['permission:PromoCodes Delete'],['only' => ['destroy']]);     
    }
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $dataTable = [];
        if ($request->ajax()) {
            $dataTable = DataTables::eloquent(
                    Promocode::query()
                    )
                    ->addIndexColumn()
                    ->editColumn('promocode_name', function(Promocode $promocode) {
                        return $promocode->promocode_name;
                    })
                    ->editColumn('discount', function(Promocode $promocode) {
                        return $promocode->discount;
                    })
                    ->editColumn('discount_type', function (Promocode $promocode) {
                        if ($promocode->discount_type == 1) {
                            return 'Flat';
                        }
    
                        return 'Percentage';
                    })    
                    ->editColumn('is_active', function(Promocode $promocode) {
                        if ($promocode->is_active) {
                            return '<a href="javascript:;" class="js-active" data-id="'.$promocode->getKey().'">
                                    <span class="px-4 lg:py-2 py-2 text-sm font-medium leading-5 text-center text-green-700 transition-colors duration-150 bg-white border border-green-700 rounded-md active:bg-green-700 hover:text-white hover:bg-green-700 focus:outline-none focus:shadow-outline-purple">Active</span>
                                </a>';
                        }

                        return '<a href="javascript:;" class="js-active" data-id="'.$promocode->getKey().'">
                                    <span class="px-4 lg:py-2 py-2 text-sm font-medium leading-5 text-center text-red-700 transition-colors duration-150 bg-white border border-red-700 rounded-md active:bg-red-700 hover:text-white hover:bg-red-700 focus:outline-none focus:shadow-outline-purple">Inactive</span>
                                </a>';
                    })
                    ->filterColumn('is_active', function($query, $keyword) {
                        $query->where('is_active', $keyword === "true");
                    })
                   
                    ->addColumn('action', function(Promocode $promocode) {
                        return '
                        <div class="flex items-center justify-right ml-auto">
                        <a href="javascript:;" data-content-page-id="'.$promocode->id.'"
                        class="ml-2 h-8 w-8 text-center flex items-center font-semibold leading-tight rounded-md text-xs text-orange-700 bg-orange-100 cursor-pointer hover:text-white hover:bg-orange-700 transition-all ease-in-out duration-300">
                        <i class="fa-solid fa-trash-can mx-auto"></i></a>
                        </div>
    
                        ';
                    })
                    ->rawColumns(['is_active', 'action', 'discount','promocode_name'])
                    ->make(true);
                    return $dataTable;
        }

        return Inertia::render('Promocode', [
            'data' => $dataTable
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return Inertia::render('AddPromocode', [
            'isCreate' => true
        ]);

    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(AddEditPromocodeRequest $request)
    {
        Promocode::create([
            'promocode_name' => $request->promocode_name,
            'discount' => $request->discount,
            'discount_type' => $request->discount_type,
            'start_date' => $request->start_date[0],
            'end_date' => $request->end_date[0],
            'is_active' => '1',
        ]);
        return Inertia::location(route('admin.promocodes.index'));

    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Promocode $promocode)
    {
        return Inertia::render('AddPromocode', [
            'promocode' => $promocode,
            'isCreate' => false
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(AddEditPromocodeRequest $request, Promocode $promocode)
    {
        $promocode->update([
            'promocode_name' => $request->promocode_name,
            'discount' => $request->discount,
            'discount_type' => $request->discount_type,
            'start_date' => $request->start_date,
            'end_date' => $request->end_date
        ]);
        return Inertia::location(route('admin.promocodes.index'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        Promocode::destroy($id);
        sleep(1);
        session()->flash('success', __('airtransport.promocode.deletesuccess'));
        return Inertia::location(route('admin.promocodes.index'));
    }

    public function active(Promocode $promocode)
    {
        $promocode->is_active = !$promocode->is_active;
        $promocode->save();

        return response()->noContent();
    }
}
