<?php

namespace App\Http\Controllers\Admin;

use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Http\Controllers\Controller;
use Spatie\Permission\Models\Permission;
use Yajra\DataTables\Facades\DataTables;
use Inertia\Inertia;

class PermissionController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $dataTable = [];
        if ($request->ajax()) {
            $dataTable = DataTables::eloquent(
                        Permission::query()
                    )
                    ->addIndexColumn()
                    ->editColumn('created_at', function (Permission $permission) {
                        return [
                            'display' => $permission->created_at->format('F j, Y'),
                            'timestamp' => $permission->created_at->timestamp
                        ];
                    })
                    ->editColumn('updated_at', function (Permission $permission) {
                        return [
                            'display' => $permission->updated_at->format('F j, Y'),
                            'timestamp' => $permission->updated_at->timestamp
                        ];
                    })
                    ->addColumn('action', function(Permission $permission) {
                        return '
                        <div class="flex items-center ml-auto">
                            <a href="'.route('admin.permissions.edit', $permission->id).'" 
                                class="h-8 w-8 text-center flex items-center font-semibold leading-tight rounded-md text-xs text-red-700 bg-orange-100 cursor-pointer hover:text-white hover:bg-orange-700 transition-all ease-in-out duration-300">
                                <i class="fa-solid fa-pencil m-auto"></i>
                            </a>
                            <a href="javascript:;" data-permission-id="'.$permission->id.'"
                            class="ml-2 h-8 w-8 text-center flex items-center font-semibold leading-tight rounded-md text-xs text-red-700 bg-red-100 cursor-pointer hover:text-white hover:bg-red-700 transition-all ease-in-out duration-300">
                            <i class="fa-solid fa-trash mx-auto"></i></a>
                        </div>
                        ';
                    })
                    ->rawColumns(['action'])
                    ->make(true);
                    return $dataTable;
        }

        // return view('admin.permissions.index');
        return Inertia::render('Permission', [
            'permission_data' => $dataTable,
            'isCreate' => true
        ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return Inertia::render('AddPermission', ['isCreate' => true]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'name'=> [
                'required',
                'min:4',
                Rule::unique('permissions', 'name'),
            ]
        ]);

        Permission::create([
            'name'=> strtolower($request->name)
        ]);
        sleep(1);
        session()->flash('success', 'Permission added successfully');
        return Inertia::location(route('admin.permissions.index'));
    }

    /**
     * Display the specified resource.
     *
     * @param  Spatie\Permission\Models\Permission  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        return view('admin.permissions.show');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  Spatie\Permission\Models\Permission  $id
     * @return \Illuminate\Http\Response
     */
    public function edit(Permission $permission)
    {
        return Inertia::render('AddPermission', [
            'isCreate'=> false, 
            'permission'=> $permission
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'name'=> [
                'required',
                'min:4',
                Rule::unique('permissions', 'name')->ignore($id),
            ]
        ]);

        Permission::where('id', $id)
        ->update([
            'name'=> strtolower($request->name)
        ]);
        sleep(1);
        session()->flash('success', 'Permission updated successfully');
        return Inertia::location(route('admin.permissions.index'));
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  Spatie\Permission\Models\Permission  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        Permission::destroy($id);
        sleep(1);
        session()->flash('success', 'Permission has been deleted successfully.');
        return Inertia::location(route('admin.permissions.index'));
    }

    public function active(Permission $permission)
    {
        $permission->is_active = !$permission->is_active;
        $permission->save();

        return response()->noContent();
    }

    public function permissionsList(Request $request)
    {

        $permissions = Permission::when($request->filled('search'), function ($query) use ($request) {
                    $query->where('name', 'like', '%' . $request->search . '%');
                })
                ->paginate(10);
        
        foreach ($permissions as $country) {
            $resultToReturn[] = [
                'id' => $country->getKey(),
                'text' => $country->name,
            ];
        }

        return response()->json([
            'results' => $resultToReturn ?? [],
            'pagination' => [
                'more' => $permissions->hasMorePages()
            ]
        ]);
    }
}
