<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AirTransportation;
use App\Models\Booking;
use App\Models\BookingPayment;
use App\Models\Driver;
use App\Models\UserSubscription;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;
use Yajra\DataTables\Facades\DataTables;

class EarningController extends Controller
{

    public function __construct()
    {
        $this->middleware(['permission:Earning View'],['only' => ['index']]);     
    }

    public function index(Request $request)
    {
        $dataTable = [];
        $total_bookings = Booking::count();
        $total_admin_earnings = UserSubscription::sum('plan_price');
        $total_driver_earnings = BookingPayment::sum('payment_amount');
        $type = isset($request->type) ? $request->type : 2;
        $filter_type = isset($request->filter_type) ? $request->filter_type : 1;
        if ($type == 2) {
            $data = $this->getDriverEarningHistory($filter_type);
        } else {
            $data = $this->getAdminEarningHistory($filter_type);
        }

        // if ($request->ajax()) {
        // //    dd($this->getDriverEarningList(3));
        //     return $this->getDriverEarningList(3);
        // }
        return Inertia::render('Earning', [
            'total_bookings' => $total_bookings,
            'total_admin_earnings' => $total_admin_earnings,
            'total_driver_earnings' => $total_driver_earnings,
            'graph_data' => $data,
            'list_data' => $dataTable,
        ]);
    }

    public function getList(Request $request)
    {
        $dataTable = [];
        if ($request->ajax()) {
            $booking = DB::table('booking_payments')
            ->select('bookings.booking_no', 'users.name as user_name', 'drivers.full_name as driver_name', 'booking_payments.payment_amount as amount', 'booking_payments.created_at as datetime', 'booking_payments.payment_amount as amount', 'booking_payments.payment_status as status')
            ->leftJoin('bookings', 'booking_payments.booking_id', '=', 'bookings.id')
            ->leftJoin('users', 'bookings.user_id', '=', 'users.id')
            ->leftJoin('drivers', 'bookings.driver_id', '=', 'drivers.id');
            $dataTable = DataTables::of($booking)
                ->addIndexColumn()
                ->editColumn('booking_no', function ($booking) {
                    return '#'.$booking->booking_no ?? "";
                })
                ->addColumn('user_name', function ($booking) {
                    return $booking->user_name ?? "";
                })
                ->addColumn('driver_name', function ($booking) {
                    $driver_name = "";
                    if ($booking->driver_name != null) {
                        $driver_name = $booking->driver_name;
                    }
                    return $driver_name;
                })
                ->addColumn('datetime', function ($booking) {
                    return $booking->datetime ?? "";
                })
                ->addColumn('amount', function ($booking) {
                    return $booking->amount ?? "";
                })

                ->addColumn('status', function ($booking) {
                    if ($booking->status == 2) {
                        return '<a href="javascript:;">
                                    <span class="px-4 lg:py-2 py-2 text-sm font-medium leading-5 text-center text-green-700 transition-colors duration-150 bg-white border border-green-700 rounded-md active:bg-green-700 hover:text-white hover:bg-green-700 focus:outline-none focus:shadow-outline-purple">PAID</span>
                                </a>';
                    } else {
                        return '<a href="javascript:;">
                        <span class="px-4 lg:py-2 py-2 text-sm font-medium leading-5 text-center text-orange-700 transition-colors duration-150 bg-white border border-orange-700 rounded-md active:bg-orange-700 hover:text-white hover:bg-orange-700 focus:outline-none focus:shadow-outline-purple">UNPAID</span>
                    </a>';
                    }
                })

                ->filterColumn('user_name', function ($booking, $keyword) {

                    $booking->where('users.name', 'LIKE', '%' . $keyword . '%');
                })
                ->filterColumn('driver_name', function ($booking, $keyword) {

                    $booking->where('drivers.full_name', 'LIKE', '%' . $keyword . '%');
                })
                ->filterColumn('datetime', function ($booking, $keyword) {
                    $dates = explode('TO', $keyword);
                    if (count($dates) == 2) {
                        [$startDate, $endDate] = explode('TO', $keyword);

                        $startDate = Carbon::createFromFormat('Y-m-d', $startDate)->startOfDay();
                        $endDate = Carbon::createFromFormat('Y-m-d', $endDate)->endOfDay();
                        $booking->whereBetween('booking_payments.created_at', [$startDate, $endDate]);
                    }
                })
                ->rawColumns(['action', 'driver_name', 'status'])
                ->make(true);
            return $dataTable;
        }
        return Inertia::render('Earning', [
            'list_data' => $dataTable
        ]);
    }

    public function getAdminList(Request $request)
    {
        $dataTable = [];
        if ($request->ajax()) {
            $booking = UserSubscription::with('subscribable')
            ->where('subscribable_type', 'App\Models\AirTransportation')
            ->orWhere('subscribable_type', 'App\Models\Driver');
            $dataTable = DataTables::of($booking)
                ->addIndexColumn()
                ->addColumn('user_name', function ($booking) {
                    $booking = $booking->subscribable;
                    $user_name = "";
                    if ($booking instanceof Driver) {
                        $user_name = $booking->full_name;
                    }
                    elseif ($booking instanceof AirTransportation) {
                        $user_name = $booking->first_name; // Assuming 'name' is the attribute in AirTransportation model
                    }
                    return $user_name;
                })
                
                ->addColumn('type', function ($booking) {
                    $type = "";
                    $booking = $booking->subscribable;
                    if ($booking instanceof Driver) {
                        $type = $booking->full_name;
                    }
                    elseif ($booking instanceof AirTransportation) {
                        $type = $booking->first_name; // Assuming 'name' is the attribute in AirTransportation model
                    }

                    return $type;
                })
               
                ->addColumn('datetime', function ($booking) {
                    return $booking->created_at ?? "";
                })
                ->addColumn('amount', function ($booking) {
                    return $booking->plan_price ?? "";
                })

               
                ->filterColumn('datetime', function ($booking, $keyword) {
                    $dates = explode('TO', $keyword);
                    if (count($dates) == 2) {
                        [$startDate, $endDate] = explode('TO', $keyword);

                        $startDate = Carbon::createFromFormat('Y-m-d', $startDate)->startOfDay();
                        $endDate = Carbon::createFromFormat('Y-m-d', $endDate)->endOfDay();
                        $booking->whereBetween('booking_payments.created_at', [$startDate, $endDate]);
                    }
                })
                ->rawColumns(['action', 'user_name', 'type'])
                ->make(true);
            return $dataTable;
        }
        return Inertia::render('Earning', [
            'list_data' => $dataTable
        ]);
    }

    private function getDriverEarningHistory($filter_type)
    {
        // $driverId = auth()->user()->id;
        $type = $filter_type;
        $data = getdefaultGrathData($type);
        $dateFormate = config("constant.charts_enum")['mysql'][$type];
        // dd($dateFormate);
        if ($type == 1) {
            $start = Carbon::now()->startOfWeek()->format("Y-m-d");
        } elseif ($type == 2) {
            $start = Carbon::now()->startOfMonth()->format("Y-m-d");
        } elseif ($type == 3) {
            $start = Carbon::now()->startOfYear()->format("Y-m-d");
        } else {
            $start = Carbon::now()->startOfWeek()->format("Y-m-d");
        }
        $graphData = DB::table('bookings')
            ->leftJoin('booking_payments', 'bookings.id', '=', 'booking_payments.booking_id')
            // ->where('bookings.driver_id', $driverId)
            ->where('booking_payments.created_at', '>=', $start)
            ->when($type == 2, function ($q) use ($dateFormate) {
                $q->select(DB::raw("CONCAT('W-', $dateFormate) as date"));
            })
            ->when($type != 2, function ($q)  use ($dateFormate) {
                $q->select(DB::raw("DATE_FORMAT(booking_payments.created_at,'$dateFormate') as date"));
            })
            ->addSelect(DB::raw("IFNULL(SUM(booking_payments.payment_amount),0) as totalAmount"))
            ->groupBy('date')
            ->orderBy('booking_payments.created_at', 'desc')
            ->get();
        // dd($graphData->toArray());
        $final_data = $data->map(function ($dataa) use ($graphData) {
            if ($graphData->contains('date', $dataa)) {
                $datails = $graphData->firstWhere('date', $dataa);
                return collect([
                    "key" => $datails->date,
                    "value" => (float)$datails->totalAmount,
                ]);
            } else {
                return collect([
                    "key" => $dataa,
                    "value" => 0
                ]);
            }
        })->values()->toArray();
        $result['list'] = $final_data;
        $result['grand_total_amount'] = (float)$graphData->sum('totalAmount');
        $totalEarningAmount = DB::table('bookings')
            ->leftJoin('booking_payments', 'bookings.id', '=', 'booking_payments.booking_id')
            ->select(DB::raw("IFNULL(SUM(booking_payments.payment_amount),0) as totalAmount"))
            // ->where('bookings.driver_id', $driverId)
            ->value('totalAmount');
        $result['total_received_amount'] = (float)$totalEarningAmount;
        return $result;
    }

    private function getAdminEarningHistory($filter_type)
    {
        // $driverId = auth()->user()->id;
        $type = $filter_type;
        $data = getdefaultGrathData($type);
        $dateFormate = config("constant.charts_enum_admin")['mysql'][$type];
        // dd($dateFormate);
        if ($type == 1) {
            $start = Carbon::now()->startOfWeek()->format("Y-m-d");
        } elseif ($type == 2) {
            $start = Carbon::now()->startOfMonth()->format("Y-m-d");
        } elseif ($type == 3) {
            $start = Carbon::now()->startOfYear()->format("Y-m-d");
        } else {
            $start = Carbon::now()->startOfWeek()->format("Y-m-d");
        }
        $graphData = DB::table('user_subscriptions')
            ->where('created_at', '>=', $start)
            ->when($type == 2, function ($q) use ($dateFormate) {
                $q->select(DB::raw("CONCAT('W-', $dateFormate) as date"));
            })
            ->when($type != 2, function ($q)  use ($dateFormate) {
                $q->select(DB::raw("DATE_FORMAT(created_at,'$dateFormate') as date"));
            })
            ->addSelect(DB::raw("IFNULL(SUM(plan_price),0) as totalAmount"))
            ->groupBy('date')
            ->orderBy('created_at', 'desc')
            ->get();
        // dd($graphData->toArray());
        $final_data = $data->map(function ($dataa) use ($graphData) {
            if ($graphData->contains('date', $dataa)) {
                $datails = $graphData->firstWhere('date', $dataa);
                return collect([
                    "key" => $datails->date,
                    "value" => (float)$datails->totalAmount,
                ]);
            } else {
                return collect([
                    "key" => $dataa,
                    "value" => 0
                ]);
            }
        })->values()->toArray();
        $result['list'] = $final_data;
        $result['grand_total_amount'] = (float)$graphData->sum('totalAmount');
        $totalEarningAmount = DB::table('user_subscriptions')
            ->select(DB::raw("IFNULL(SUM(plan_price),0) as totalAmount"))
            // ->where('bookings.driver_id', $driverId)
            ->value('totalAmount');
        $result['total_received_amount'] = (float)$totalEarningAmount;
        return $result;
    }
}
