<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Driver;
use App\Notifications\DriverApprovalStatusNotification;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Carbon\Carbon;
use Yajra\DataTables\Facades\DataTables;

class DriverController extends Controller
{

    public function __construct()
    {
        $this->middleware(['permission:Driver List|Driver View|Driver Update'], ['only' => ['index']]);
        $this->middleware(['permission:Driver View'], ['only' => ['show']]);
        $this->middleware(['permission:Driver Update'], ['only' => ['active', 'occupy', 'updateStatus']]);
    }
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $dataTable = DataTables::eloquent(
                Driver::query()
            )
                ->addIndexColumn()
                ->addColumn('total_rides', function (Driver $driver) {
                    $totalRides = $driver->bookings->count();

                    return $totalRides;
                })
                ->addColumn('approval_status', function (Driver $driver) {
                    if ($driver->is_approved == 0) {
                        return '<select class="block w-full rounded-md border-0 py-1.5 px-3 text-gray-900 dark:text-gray-300 shadow-sm ring-1 ring-gray-300 dark:ring-gray-600 placeholder:text-gray-400 outline-none focus:outline-0 sm:text-sm sm:leading-6 dark:bg-transparent h-[40px] bg-white update-status" data-order-id="' . $driver->id . '">
                                <option value="0" selected>Pending</option>
                                <option value="1">Approved</option>
                                <option value="2">Rejected</option>
                            </select>';
                    }
                    return $driver->is_approved == 1 ? '<a href="javascript:;">
                    <span class="px-4 lg:py-2 py-2 text-sm font-medium leading-5 text-center text-green-700 transition-colors duration-150 bg-white border border-green-700 rounded-md active:bg-green-700 hover:text-white hover:bg-green-700 focus:outline-none focus:shadow-outline-purple">Approved</span>
                </a>' : '<a href="javascript:;">
                <span class="px-4 lg:py-2 py-2 text-sm font-medium leading-5 text-center text-red-700 transition-colors duration-150 bg-white border border-red-700 rounded-md active:bg-red-700 hover:text-white hover:bg-red-700 focus:outline-none focus:shadow-outline-purple">Rejected</span>
            </a>';
                })

                ->editColumn('is_active', function (Driver $driver) {
                    if ($driver->is_active) {
                        return '<a href="javascript:;" class="js-active" data-id="' . $driver->getKey() . '">
                                    <span class="px-4 lg:py-2 py-2 text-sm font-medium leading-5 text-center text-green-700 transition-colors duration-150 bg-white border border-green-700 rounded-md active:bg-green-700 hover:text-white hover:bg-green-700 focus:outline-none focus:shadow-outline-purple">Active</span>
                                </a>';
                    }

                    return '<a href="javascript:;" class="js-active" data-id="' . $driver->getKey() . '">
                                    <span class="px-4 lg:py-2 py-2 text-sm font-medium leading-5 text-center text-red-700 transition-colors duration-150 bg-white border border-red-700 rounded-md active:bg-red-700 hover:text-white hover:bg-red-700 focus:outline-none focus:shadow-outline-purple">Inactive</span>
                                </a>';
                })
                ->filterColumn('is_active', function ($query, $keyword) {
                    $query->where('is_active', $keyword === "true");
                })
                ->filterColumn('approval_status', function ($query, $keyword) {
                    if ($keyword == -1) {
                        $query->whereIn('is_approved', [0, 1, 2]);
                    } elseif ($keyword == 1) {
                        $query->where('is_approved', 1);
                    } elseif ($keyword == 2) {
                        $query->where('is_approved', 2);
                    }
                })
                ->filterColumn('created_at', function ($query, $keyword) {
                    $dates = explode('TO', $keyword);
                    if (count($dates) == 2) {
                        [$startDate, $endDate] = explode('TO', $keyword);

                        $startDate = Carbon::createFromFormat('Y-m-d', $startDate)->startOfDay();
                        $endDate = Carbon::createFromFormat('Y-m-d', $endDate)->endOfDay();
                        $query->whereBetween('created_at', [$startDate, $endDate]);
                    }
                })
                ->filterColumn('updated_at', function ($query, $keyword) {
                    $query->whereDate('updated_at', $keyword);
                })
                ->editColumn('created_at', function ($driver) {
                    return [
                        'display' => $driver->created_at->format('F j, Y'),
                        'timestamp' => $driver->created_at->timestamp
                    ];
                })
                ->editColumn('updated_at', function ($driver) {
                    return [
                        'display' => $driver->updated_at->format('F j, Y'),
                        'timestamp' => $driver->updated_at->timestamp
                    ];
                })
                ->addColumn('action', function (Driver $driver) {
                    $element = '<div class="flex items-center ml-auto">
                                <a href="' . route('admin.drivers.show', $driver) . '" 
                                class="mx-2 ml-2 h-8 w-8 text-center flex items-center font-semibold leading-tight rounded-md text-xs text-orange-700 bg-orange-100 cursor-pointer hover:text-white hover:bg-orange-700 transition-all ease-in-out duration-300">
                                    <i class="fa-solid fa-eye mx-auto"></i>
                                </a>';

                    if ($driver->is_occupied == 1) {
                        $element .= '<a href="javascript:;" class="js-occupy" data-id="' . $driver->getKey() . '">
                                        <span class="px-4 lg:py-2 py-2 text-sm font-medium leading-5 text-center text-red-700 transition-colors duration-150 bg-white border border-red-700 rounded-md active:bg-red-700 hover:text-white hover:bg-red-700 focus:outline-none focus:shadow-outline-purple">Occupied</span>
                                    </a>';
                    } else {
                        $element .= '<a href="javascript:;" class="js-occupy" data-id="' . $driver->getKey() . '">
                                        <span class="px-4 lg:py-2 py-2 text-sm font-medium leading-5 text-center text-green-700 transition-colors duration-150 bg-white border border-green-700 rounded-md active:bg-green-700 hover:text-white hover:bg-green-700 focus:outline-none focus:shadow-outline-purple">Free</span>
                                    </a>';
                    }

                    $element .= '</div>';

                    return $element;
                })
                ->rawColumns(['is_active', 'action', 'approval_status'])
                ->make(true);
            return $dataTable;
        }
        return Inertia::render('Driver', [
            'dataTable' => [],
            'activeOptions' =>  [
                [
                    "text" => "Active",
                    "value" => true
                ],
                [
                    "text" => "Inactive",
                    "value" => false
                ]
            ]
        ]);

        // return Inertia::render('Driver');
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //
    }

    /**
     * Display the specified resource.
     */
    public function show(Driver $driver)
    {
        $driver = $driver->load('vehicleinfo.car');
        // dd($driver);
        return Inertia::render('ViewDriver', [
            'driver' => $driver,
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }

    public function updateStatus(Request $request, $id)
    {
        $driver = Driver::findOrFail($id);
        $driver->update(['is_approved' => $request->status]); //'reason' => $request->reason
        $driver->notify(new DriverApprovalStatusNotification($request->reason, $request->status));
        return response()->json(['message' => 'Status updated successfully']);
    }

    public function occupy(Driver $driver)
    {
        $driver->is_occupied = !$driver->is_occupied;
        $driver->save();

        return response()->noContent();
    }

    public function active(Driver $driver)
    {
        $driver->is_active = !$driver->is_active;
        $driver->save();
        if (!$driver->is_active) {
            $driver->tokens()->delete();
        }


        return response()->noContent();
    }
}
