<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Car;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Inertia\Inertia;
use Yajra\DataTables\Facades\DataTables;

class CarController extends Controller
{
    public function __construct()
    {
        $this->middleware(['permission:Car List|Car Update|Car Delete|Car Add'],['only' => ['index']]);     
        $this->middleware(['permission:Car Update'],['only' => ['edit','update','active']]);     
        $this->middleware(['permission:Car Add'],['only' => ['create','store']]);     
        $this->middleware(['permission:Car Delete'],['only' => ['destroy']]);     
    }
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $dataTable = [];
        if ($request->ajax()) {
            $url = config('services.aws.aws_cloud_url');
            $dataTable = DataTables::eloquent(
                        Car::query()
                    )
                    ->addIndexColumn()
                    ->editColumn('car_image', function(Car $car) use($url){
                        return '<img class="w-12 h-12 border rounded-full" src="'.$url.$car->car_image.'" />';
                    })
                    ->editColumn('is_active', function(Car $car) {
                        if ($car->is_active) {
                            return '<a href="javascript:;" class="js-active" data-id="'.$car->getKey().'">
                            <span class="px-4 lg:py-2 py-2 text-sm font-medium leading-5 text-center text-green-700 transition-colors duration-150 bg-white border border-green-700 rounded-md active:bg-green-700 hover:text-white hover:bg-green-700 focus:outline-none focus:shadow-outline-purple">Active</span>
                                </a>';
                        }

                        return '<a href="javascript:;" class="js-active" data-id="'.$car->getKey().'">
                        <span class="px-4 lg:py-2 py-2 text-sm font-medium leading-5 text-center text-red-700 transition-colors duration-150 bg-white border border-red-700 rounded-md active:bg-red-700 hover:text-white hover:bg-red-700 focus:outline-none focus:shadow-outline-purple">Inactive</span>
                                </a>';
                    })
                    ->editColumn('updated_at', function (Car $car) {
                        return [
                            'display' => $car->updated_at->format('F j, Y'),
                            'timestamp' => $car->updated_at->timestamp
                        ];
                    })
                    ->addColumn('action', function(Car $car) {
                        return '
                        <div class="flex items-center ml-auto">
                        
                    <a href="'.route('admin.cars.edit', $car).'" 
                        class="ml-2 h-8 w-8 text-center flex items-center font-semibold leading-tight rounded-md text-xs text-red-700 bg-red-100 cursor-pointer hover:text-white hover:bg-red-700 transition-all ease-in-out duration-300">
                        <i class="fa-solid fa-pencil m-auto"></i>
                    </a>
                    <a href="javascript:;" data-car-id="'.$car->id.'"
                    class="ml-2 h-8 w-8 text-center flex items-center font-semibold leading-tight rounded-md text-xs text-orange-700 bg-orange-100 cursor-pointer hover:text-white hover:bg-orange-700 transition-all ease-in-out duration-300">
                    <i class="fa-solid fa-trash mx-auto"></i></a>
                    </div>

                        ';
                    })
                    ->rawColumns(['is_active', 'action', 'car_image'])
                    ->make(true);
                    return $dataTable;
        }
        return Inertia::render('Car', [
            'data' => $dataTable, // Initialize with empty array
        ]);

    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return Inertia::render('AddCar', [
            'isCreate' => true,
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'car_type_name' => 'required|string',
            'car_description' => 'required',
            'car_capacity' => 'required|integer',
            'car_image' => 'required',
            
        ]);
            
        $file = $request->file('car_image');
        $fileName = time()."-".$request->car_image->getClientOriginalName();
        $filePath = 'uploads/carimages/' . $fileName;
        $path = Storage::disk('s3')->put($filePath, file_get_contents($file));
        $url = Storage::disk('s3')->url($filePath);

        Car::create([
            'uuid' => GetUuid(),
            'car_type_name' => $request->car_type_name,
            'car_description' => $request->car_description,
            'car_capacity' => $request->car_capacity,
            'car_image' => $filePath,
            
        ]);
        sleep(1);
        session()->flash('success', 'Car added successfully');
        return Inertia::location(route('admin.cars.index'));
        // return redirect()->route('admin.cars.index')->with('success','Car added successfully');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Car $car)
    {
        return Inertia::render('AddCar', [
            'cardata' => $car,
            'isCreate' => false,
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'car_type_name' => 'required|string',
            'car_description' => 'required',
            'car_capacity' => 'required|integer',
            'car_image' => 'required',
            
        ]);
            
        $file = $request->file('car_image');
        $fileName = time()."-".$request->car_image->getClientOriginalName();
        $filePath = 'uploads/carimages/' . $fileName;
        $path = Storage::disk('s3')->put($filePath, file_get_contents($file));
        $url = Storage::disk('s3')->url($filePath);

        Car::where('id', $id)->update([
            'uuid' => GetUuid(),
            'car_type_name' => $request->car_type_name,
            'car_description' => $request->car_description,
            'car_capacity' => $request->car_capacity,
            'car_image' => $filePath,
            
        ]);
        sleep(1);
        session()->flash('success', 'Car updated successfully');
        return Inertia::location(route('admin.cars.index'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Car $car)
    {
        // $carDelete = Car::findOrFail($id);
            $car->delete();
            sleep(1);
        return redirect()->route('admin.cars.index')->with('success','Car deleted successfully');
    }

    public function active(Car $car)
    {
        $car->is_active = !$car->is_active;
        $car->save();

        return response()->noContent();
    }

}
