<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AirTransportationBooking;
use App\Models\Booking;
use App\Models\Driver;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;
use Yajra\DataTables\Facades\DataTables;


class BookingController extends Controller
{

    
    public function __construct()
    {
        $this->middleware(['permission:Bookings List|Bookings View|Bookings Update'],['only' => ['index','bookings','airBooking']]);     
        $this->middleware(['permission:Bookings View'],['only' => ['show','showAirBooking']]);    
        $this->middleware(['permission:Bookings Update'],['only' => ['getDrivers','assignDriver']]); 
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $dataTable = [];
        if ($request->ajax()) {
            $booking = DB::table('bookings as booking')
                ->select('booking.id as id', 'booking.booking_no', 'users.name as customer_name', 'booking_addresses.pickup_location', 'booking_addresses.dropoff_location', 'drivers.full_name as driver_name', 'booking.booking_date_time as booking_date', 'booking.booking_status as status')
                ->join('users', 'booking.user_id', '=', 'users.id')
                ->leftJoin('drivers', 'booking.driver_id', '=', 'drivers.id')
                ->leftJoin('booking_addresses', 'booking.id', '=', 'booking_addresses.booking_id')
                ->where('admin_assign', 1);
            $dataTable = DataTables::of($booking)
                ->addIndexColumn()
                ->editColumn('booking_no', function ($booking) {
                    return '#' . $booking->booking_no ?? "";
                })
                ->editColumn('pickup_location', function ($booking) {
                    return '<p style="max-width: 300px;
                    white-space: normal;
                    word-break: break-word;"> ' . $booking->pickup_location . '</p>';
                })
                ->editColumn('dropoff_location', function ($booking) {
                    return '<p style="max-width: 300px;
                    white-space: normal;
                    word-break: break-word;"> ' . $booking->dropoff_location . '</p>';
                })
                ->editColumn('driver_name', function ($booking) {
                    $driver_name = "";
                    if ($booking->driver_name != null) {
                        $driver_name = $booking->driver_name;
                    }
                    return $driver_name;
                })

                ->editColumn('status', function ($booking) {
                    if ($booking->status == 6) {
                        return '<a href="javascript:;">
                                    <span class="px-4 lg:py-2 py-2 text-sm font-medium leading-5 text-center text-green-700 transition-colors duration-150 bg-white border border-green-700 rounded-md active:bg-green-700 hover:text-white hover:bg-green-700 focus:outline-none focus:shadow-outline-purple">' . getBookingStatus($booking->status) . '</span>
                                </a>';
                    } else {
                        return '<a href="javascript:;">
                        <span class="px-4 lg:py-2 py-2 text-sm font-medium leading-5 text-center text-orange-700 transition-colors duration-150 bg-white border border-orange-700 rounded-md active:bg-orange-700 hover:text-white hover:bg-orange-700 focus:outline-none focus:shadow-outline-purple">' . getBookingStatus($booking->status) . '</span>
                    </a>';
                    }
                })

                ->filterColumn('customer_name', function ($booking, $keyword) {

                    $booking->where('users.name', 'LIKE', '%' . $keyword . '%');
                })
                ->filterColumn('driver_name', function ($booking, $keyword) {

                    $booking->where('drivers.full_name', 'LIKE', '%' . $keyword . '%');
                })
                ->filterColumn('booking_date', function ($booking, $keyword) {
                    $dates = explode('TO', $keyword);
                    if (count($dates) == 2) {
                        [$startDate, $endDate] = explode('TO', $keyword);

                        $startDate = Carbon::createFromFormat('Y-m-d', $startDate)->startOfDay();
                        $endDate = Carbon::createFromFormat('Y-m-d', $endDate)->endOfDay();
                        $booking->whereBetween('booking.booking_date_time', [$startDate, $endDate]);
                    }
                })
                ->filterColumn('pickup_location', function ($booking, $keyword) {

                    $booking->where('booking_addresses.pickup_location', 'LIKE', '%' . $keyword . '%');
                })
                ->filterColumn('dropoff_location', function ($booking, $keyword) {

                    $booking->where('booking_addresses.dropoff_location', 'LIKE', '%' . $keyword . '%');
                })

                ->addColumn('action', function ($booking) {
                    return '
                    <div class="flex items-center justify-center ml-auto">
                    <a class="h-8 w-8 text-center flex items-center font-semibold leading-tight rounded-md text-xs text-green-700 bg-green-100 cursor-pointer hover:text-white hover:bg-green-700 transition-all ease-in-out duration-300"
                        href="' . route('admin.bookings.show', $booking->id) . '">
                        <i class="fa-solid fa-eye mx-auto"></i>
                    </a>
                    <a title="Assign Driver" data-booking-id="' . $booking->id . '"
                        class="ml-2 h-8 w-8 text-center flex items-center font-semibold leading-tight rounded-md text-xs text-orange-700 bg-orange-100 cursor-pointer hover:text-white hover:bg-orange-700 transition-all ease-in-out duration-300 assign-driver"
                        href="#">
                        <i class="fa-solid fa-user m-auto"></i>
                    </a>
                </div>

                        ';
                })
                ->rawColumns(['action', 'driver_name', 'status', 'pickup_location', 'dropoff_location', 'booking_no'])
                ->make(true);
            return $dataTable;
        }

        return Inertia::render('Booking', [
            'data' => $dataTable
        ]);
    }

    public function bookings(Request $request)
    {
        $dataTable = [];
        if ($request->ajax()) {
            $booking = DB::table('bookings as booking')
                ->select('booking.id as id', 'booking.booking_no', 'users.name as customer_name', 'booking_addresses.pickup_location', 'booking_addresses.dropoff_location', 'drivers.full_name as driver_name', 'booking.booking_date_time as booking_date', 'booking.booking_status as status')
                ->join('users', 'booking.user_id', '=', 'users.id')
                ->leftJoin('drivers', 'booking.driver_id', '=', 'drivers.id')
                ->leftJoin('booking_addresses', 'booking.id', '=', 'booking_addresses.booking_id');
            $dataTable = DataTables::of($booking)
                ->addIndexColumn()
                ->editColumn('booking_no', function ($booking) {
                    return '#' . $booking->booking_no ?? "";
                })
                ->editColumn('pickup_location', function ($booking) {
                    return '<p style="max-width: 300px;
                    white-space: normal;
                    word-break: break-word;"> ' . $booking->pickup_location . '</p>';
                })
                ->editColumn('dropoff_location', function ($booking) {
                    return '<p style="max-width: 300px;
                    white-space: normal;
                    word-break: break-word;"> ' . $booking->dropoff_location . '</p>';
                })
                ->editColumn('driver_name', function ($booking) {
                    $driver_name = "";
                    if ($booking->driver_name != null) {
                        $driver_name = $booking->driver_name;
                    }
                    return $driver_name;
                })

                ->editColumn('status', function ($booking) {
                    if ($booking->status == 6) {
                        return '<a href="javascript:;">
                                    <span class="px-4 lg:py-2 py-2 text-sm font-medium leading-5 text-center text-green-700 transition-colors duration-150 bg-white border border-green-700 rounded-md active:bg-green-700 hover:text-white hover:bg-green-700 focus:outline-none focus:shadow-outline-purple">' . getBookingStatus($booking->status) . '</span>
                                </a>';
                    } else {
                        return '<a href="javascript:;">
                        <span class="px-4 lg:py-2 py-2 text-sm font-medium leading-5 text-center text-orange-700 transition-colors duration-150 bg-white border border-orange-700 rounded-md active:bg-orange-700 hover:text-white hover:bg-orange-700 focus:outline-none focus:shadow-outline-purple">' . getBookingStatus($booking->status) . '</span>
                    </a>';
                    }
                })

                ->filterColumn('customer_name', function ($booking, $keyword) {

                    $booking->where('users.name', 'LIKE', '%' . $keyword . '%');
                })

                ->filterColumn('driver_name', function ($booking, $keyword) {

                    $booking->where('drivers.full_name', 'LIKE', '%' . $keyword . '%');
                })
                ->filterColumn('booking_date', function ($booking, $keyword) {
                    $dates = explode('TO', $keyword);
                    if (count($dates) == 2) {
                        [$startDate, $endDate] = explode('TO', $keyword);

                        $startDate = Carbon::createFromFormat('Y-m-d', $startDate)->startOfDay();
                        $endDate = Carbon::createFromFormat('Y-m-d', $endDate)->endOfDay();
                        $booking->whereBetween('booking.booking_date_time', [$startDate, $endDate]);
                    }
                })
                ->filterColumn('pickup_location', function ($booking, $keyword) {

                    $booking->where('booking_addresses.pickup_location', 'LIKE', '%' . $keyword . '%');
                })
                ->filterColumn('dropoff_location', function ($booking, $keyword) {

                    $booking->where('booking_addresses.dropoff_location', 'LIKE', '%' . $keyword . '%');
                })

                ->addColumn('action', function ($booking) {
                    return '
                    <div class="flex items-center justify-center ml-auto">
                    <a class="h-8 w-8 text-center flex items-center font-semibold leading-tight rounded-md text-xs text-green-700 bg-green-100 cursor-pointer hover:text-white hover:bg-green-700 transition-all ease-in-out duration-300"
                        href="' . route('admin.bookings.show', $booking->id) . '">
                        <i class="fa-solid fa-eye mx-auto"></i>
                    </a>
                </div>

                        ';
                })
                ->rawColumns(['action', 'driver_name', 'status', 'pickup_location', 'dropoff_location', 'booking_no'])
                ->make(true);
            return $dataTable;
        }

        return Inertia::render('Booking', [
            'booking_data' => $dataTable
        ]);
    }

    public function airBooking(Request $request)
    {
        $dataTable = [];
        if ($request->ajax()) {
            $booking = DB::table('air_transportation_bookings as booking')
                ->select('booking.id as id', 'booking.booking_no', 'users.name as customer_name', 'booking.pickup_location', 'booking.dropoff_location', 'air_transportations.first_name', 'air_transportations.last_name', 'air_transportations.company_name', 'air_transportations.type', 'booking.created_at as booking_date', 'booking.status as status')
                ->leftJoin('air_transportations', 'booking.air_transportation_id', '=', 'air_transportations.id')
                ->join('users', 'booking.user_id', '=', 'users.id');
            $dataTable = DataTables::of($booking)
                ->addIndexColumn()
                ->editColumn('booking_no', function ($booking) {
                    return '#' . $booking->booking_no ?? "";
                })
                ->editColumn('pickup_location', function ($booking) {
                    return $booking->pickup_location ?? "";
                })
                ->editColumn('dropoff_location', function ($booking) {
                    return $booking->dropoff_location ?? "";
                })
                ->editColumn('company_name', function ($booking) {
                    $company_name = "";
                    if ($booking->type == 1) {
                        $company_name = $booking->company_name;
                    } else {
                        $company_name = $booking->first_name . " " . $booking->last_name;
                    }
                    return $company_name;
                })

                ->editColumn('status', function ($booking) {
                    if ($booking->status == 3) {
                        return '<a href="javascript:;">
                                    <span class="px-4 lg:py-2 py-2 text-sm font-medium leading-5 text-center text-green-700 transition-colors duration-150 bg-white border border-green-700 rounded-md active:bg-green-700 hover:text-white hover:bg-green-700 focus:outline-none focus:shadow-outline-purple">' . getAirBookingStatus($booking->status) . '</span>
                                </a>';
                    } else {
                        return '<a href="javascript:;">
                        <span class="px-4 lg:py-2 py-2 text-sm font-medium leading-5 text-center text-orange-700 transition-colors duration-150 bg-white border border-orange-700 rounded-md active:bg-orange-700 hover:text-white hover:bg-orange-700 focus:outline-none focus:shadow-outline-purple">' . getAirBookingStatus($booking->status) . '</span>
                    </a>';
                    }
                })




                ->addColumn('action', function ($booking) {
                    return '
                    <div class="flex items-center justify-center ml-auto">
                    <a class="h-8 w-8 text-center flex items-center font-semibold leading-tight rounded-md text-xs text-green-700 bg-green-100 cursor-pointer hover:text-white hover:bg-green-700 transition-all ease-in-out duration-300"
                        href="' . route('admin.airbookings.show', $booking->id) . '">
                        <i class="fa-solid fa-eye mx-auto"></i>
                    </a>
                </div>

                        ';
                })
                ->rawColumns(['action', 'driver_name', 'status', 'pickup_location', 'dropoff_location', 'booking_no'])
                ->make(true);
            return $dataTable;
        }

        return Inertia::render('Booking', [
            'air_booking_data' => $dataTable
        ]);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $bookingDetails = Booking::with([
            'users:id,profile_photo,name,sendbird_chat_id,mobile',
            'driver:id,profile_image,full_name,sendbird_chat_id,mobile',
            'car:id,car_type_name',
            'promocode:id,promocode_name,discount,discount_type',
            'review:id,booking_id,rating',
            'driver.vehicleinfo:driver_id,plate_number',
            'bookingaddress:booking_id,pickup_location,dropoff_location,pickup_lat,pickup_long,dropoff_lat,dropoff_long',
            'cancelbooking:booking_id,cancel_reason_id,cancel_description,cancelled_by',
            'cancelbooking.cancelreason:id,reason',
        ])->findOrFail($id);
        return Inertia::render('BookingDetail', [
            'booking_data' => $bookingDetails
        ]);
    }

    public function showAirBooking(string $id)
    {
        $bookingDetails = AirTransportationBooking::with([
            'users:id,profile_photo,name',
            'air_transportations:id,profile_image,type,first_name,last_name,company_name,sendbird_chat_id,mobile',
        ])->select([
            'id',
            'booking_no',
            'user_id',
            'air_transportation_id',
            'pickup_location',
            'dropoff_location',
            'flying_type',
            'comment',
            'adult',
            'child',
            'infants',
            'trip_type',
            'booking_price',
            'status',
            DB::raw("DATE_FORMAT(departure_date_time, '%e %M %Y, %h:%i %p') as formatted_departure_datetime"),
            DB::raw("DATE_FORMAT(arrival_date_time, '%e %M %Y, %h:%i %p') as formatted_arrival_datetime"),
            DB::raw("DATE_FORMAT(created_at, '%e %M %Y') as created_datetime")
        ])
            ->findOrFail($id);
        return Inertia::render('AirBookingDetail', [
            'booking_data' => $bookingDetails
        ]);
    }

    public function getDrivers(Request $request)
    {
        $booking = Booking::with('bookingaddress')->find($request->id);
        $pickupLat = $booking->bookingaddress->pickup_lat;
        $pickupLong = $booking->bookingaddress->pickup_long;
        $radiusMeters = 100;
        $carId = $booking->car_id;
        $driversWithinRadius = Driver::select('id', 'full_name', 'latitude', 'longitude', DB::raw("
            (3959 * acos(cos(radians($pickupLat)) * cos(radians(latitude)) *
            cos(radians(longitude) - radians($pickupLong)) +
            sin(radians($pickupLat)) * sin(radians(latitude))))
            AS distance
            "))
            ->having('distance', '<=', $radiusMeters)
            ->where('online_status', '=', 1)
            ->where('is_occupied', '=', 0)
            ->whereHas('vehicleinfo', function ($query) use ($carId) {
                $query->where('vehicle_type', $carId);
            })
            ->orderBy('distance', 'asc')
            ->get();
        $drivers = $driversWithinRadius->makeHidden(['latitude', 'longitude', 'distance']);
        // dd($drivers);
        return Inertia::render('Booking', [
            'drivers' => $drivers
        ]);
    }

    public function assignDriver(Request $request)
    {
        // dd($request);
        $booking = Booking::find($request->booking_id);
        $booking->driver_id = $request->driver_id;
        $booking->save();
        return Inertia::location(route('admin.bookings.index'));
    }
}
