<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AirTransportation;
use App\Notifications\AirTransportationStatusUpdateNotification;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Yajra\DataTables\Facades\DataTables;

class AirTransportationController extends Controller
{
    public function __construct()
    {
        $this->middleware(['permission:Air Transportation List|Air Transportation View|Air Transportation Delete|Air Transportation Update'],['only' => ['index']]);     
        $this->middleware(['permission:Air Transportation View'],['only' => ['show']]);     
        $this->middleware(['permission:Air Transportation Delete'],['only' => ['destroy']]);     
        $this->middleware(['permission:Air Transportation Update'],['only' => ['active','updateStatus']]);
    }
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $dataTable = [];
        if ($request->ajax()) {
            $airTransportation = AirTransportation::select([
                'air_transportations.*',
                'air_transportations.company_name as company',
                ])
                ->leftJoin('jets', 'jets.air_transportation_id', '=', 'air_transportations.id')
                ->selectRaw('COUNT(jets.air_transportation_id) as total_jets')
                ->groupBy('air_transportations.id')
                ->whereNull('air_transportations.deleted_at');

            $dataTable = DataTables::of($airTransportation)
                ->addIndexColumn()
                ->addColumn('type', function (AirTransportation $airTransportation) {
                    if ($airTransportation->type == 1) {
                        return 'Company';
                    }
    
                    return 'Individual';
                })
                ->editColumn('is_active', function (AirTransportation $airTransportation) {
                    if ($airTransportation->is_active) {
                        return '<a href="javascript:;" class="js-active" data-id="' . $airTransportation->getKey() . '">
                                    <span class="px-4 lg:py-2 py-2 text-sm font-medium leading-5 text-center text-green-700 transition-colors duration-150 bg-white border border-green-700 rounded-md active:bg-green-700 hover:text-white hover:bg-green-700 focus:outline-none focus:shadow-outline-purple">Active</span>
                                </a>';
                    }

                    return '<a href="javascript:;" class="js-active" data-id="' . $airTransportation->getKey() . '">
                                    <span class="px-4 lg:py-2 py-2 text-sm font-medium leading-5 text-center text-red-700 transition-colors duration-150 bg-white border border-red-700 rounded-md active:bg-red-700 hover:text-white hover:bg-red-700 focus:outline-none focus:shadow-outline-purple">Inactive</span>
                                </a>';
                })

                ->editColumn('request_status', function (AirTransportation $airTransportation) {
                    if ($airTransportation->request_status == 0) {
                        return '<select class="block w-full rounded-md border-0 py-1.5 px-3 text-gray-900 dark:text-gray-300 shadow-sm ring-1 ring-gray-300 dark:ring-gray-600 placeholder:text-gray-400 outline-none focus:outline-0 sm:text-sm sm:leading-6 dark:bg-transparent h-[40px] bg-white air-transpot-update-status" data-order-id="' . $airTransportation->id . '">
                                <option value="0" selected>Pending</option>
                                <option value="1">Accept</option>
                                <option value="2">Reject</option>
                            </select>';
                    }
                    return $airTransportation->request_status === 1 ? 'Accept' : 'Reject';
                })
                ->filterColumn('request_status', function ($query, $keyword) {
                    if ($keyword === 'All') {
                        $query->whereIn('request_status', [0, 1, 2]);
                    } elseif ($keyword === 'Accept') {
                        $query->where('request_status', 1);
                    } elseif ($keyword === 'Reject') {
                        $query->where('request_status', 2);
                    }
                })
                
                ->editColumn('created_at', function ($driver) {
                    return [
                        'display' => $driver->created_at->format('F j, Y'),
                        'timestamp' => $driver->created_at->timestamp
                    ];
                })
                ->editColumn('total_jets', function (AirTransportation $airTransportation) {
                    return $airTransportation->total_jets;
                })

                ->addColumn('action', function (AirTransportation $airTransportation) {
                    return '
                        <div class="flex items-center justify-right ml-auto">
                        
                        <a href="'.route('admin.air-transportation.show', $airTransportation->id) . '" 
                            class="h-8 w-8 text-center flex items-center font-semibold leading-tight rounded-md text-xs text-green-700 bg-green-100 cursor-pointer hover:text-white hover:bg-green-700 transition-all ease-in-out duration-300">
                            <i class="fa-solid fa-eye mx-auto"></i>
                        </a>
                        <a href="javascript:;" data-content-page-id="' . $airTransportation->id . '"
                        class="ml-2 h-8 w-8 text-center flex items-center font-semibold leading-tight rounded-md text-xs text-orange-700 bg-orange-100 cursor-pointer hover:text-white hover:bg-orange-700 transition-all ease-in-out duration-300">
                        <i class="fa-solid fa-trash-can mx-auto"></i></a>
                        </div>
    
                        ';
                })
                ->rawColumns(['is_active', 'action', 'type', 'request_status', 'total_jets'])
                ->make(true);

            return $dataTable;
        }

        return Inertia::render('AirTransportationCompany', [
            'data' => $dataTable
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //
    }

    /**
     * Display the specified resource.
     */
    public function show(AirTransportation $airTransportation)
    {
        $airTransportationWithRelatedData = AirTransportation::with(['jets', 'pilot', 'terminal'])
            ->find($airTransportation->id);
        $jetTypes = config('constant.jet_types');
        return Inertia::render('AirTransportationList', [
            'airTransportation' => $airTransportationWithRelatedData,
            'jetTypes' => $jetTypes,
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        AirTransportation::destroy($id);
        session()->flash('success', __('airtransport.air.deletesuccess'));
        return Inertia::location(route('admin.air-transportation.index'));
    }

    public function active(AirTransportation $air_transportation)
    {
        $air_transportation->is_active = !$air_transportation->is_active;
        $air_transportation->save();
        session()->flash('message', __('airtransport.reviews.activestatus'));
        return response()->noContent();
    }

    public function updateStatus(Request $request, AirTransportation $air_transportation)
    {
        $user = auth()->user();
        $air_transportation->update(['request_status' => $request->input('request_status'), 'reason' => $request->input('reason')]);
        $user->notify(new AirTransportationStatusUpdateNotification($request->input('reason'), $request->input('request_status')));
        session()->flash('success', __('airtransport.air.email'));
        return response()->noContent();
    }
}
