<?php

namespace App\Http\Controllers\APIs\v1;

use App\Models\SocialLogin;
use SpaceO\RESTAuth\Traits\UserProfileTrait;
use SpaceO\RESTAuth\Traits\SendOTPTrait;
use SpaceO\RESTAuth\Http\Controllers\Controller;
use App\Models\User;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;
use Twilio\Rest\Client;

class UserProfileController extends Controller
{

    use UserProfileTrait;
    use SendOTPTrait;
    private int $otpLength;

    public function __construct()
    {
        $this->otpLength = config('rest_auth.otp_length');
    }

    public function edit(Request $request)
    {
        try {
            $user = User::findOrFail(auth()->user()->id);
            if (isset($request->email)) {
                if ($user->email != $request->email) {
                    $request->validate([
                        'email' => [
                            'string',
                            'email',
                            'max:255',
                            Rule::unique('users', 'email')
                        ],
                    ]);
                }
                $user->email = $request->email;
            }
            if (isset($request->name)) {
                $user->name = $request->name;
            }
            if (isset($request->profile_photo)) {
                if (config('services.aws.aws_cloud_url') . $user->profile_photo != $request->profile_photo) {
                    $user->profile_photo = $request->profile_photo;
                }
            }
            if (isset($request->isd_code)) {
                $user->isd_code = $request->isd_code;
            }
            if (isset($request->mobile) && ($user->mobile != $request->mobile)) {
                $request->merge([
                    'usermobile' => $request->isd_code . $request->mobile,
                ]);
                $request->validate(
                    [
                        'mobile' => Rule::unique('users', 'mobile'),
                        'usermobile' => 'phone:AUTO',
                    ],
                    [
                        'usermobile.phone' => 'Please enter a valid mobile number.', // custom message
                    ]
                );
                $user->mobile = $request->mobile;
                $user->mobile_verified_at = null;
                $otp = $this->sendOTPOnMobile($request);
                User::where('id', $user['id'])->update(
                    array_combine($this->OTPFields(), [
                        $otp, now()->addMinutes(config('rest_auth.otp_expires_in'))
                    ])
                );
            }
            $user->save();
            $user = $user->only(['id', 'name', 'email', 'isd_code', 'mobile', 'profile_photo', 'mobile_verified_at', 'stripe_id', 'sendbird_chat_id', 'is_social_account']);
            if ($user['mobile_verified_at'] == null) {
                $user['is_verified'] = 0;
            } else {
                $user['is_verified'] = 1;
            }
            if ($user['is_social_account'] == 0) {
                $user['login_type'] = 0;
                $user['social_id'] = "";
            } else {
                $social_details = SocialLogin::where('user_id', $user['id'])->first();
                $user['login_type'] = $social_details->type;
                $user['social_id'] = $social_details->social_id;
            }

            $user['rating'] = "0.0";
            $user['is_subscribed'] = 0;
            $user['is_profile_completed'] = 0;
            $user['step'] = 0;
            $user['profile_image'] = $user['profile_photo'] == null ? '' : (config('services.aws.aws_cloud_url') . $user['profile_photo']);
            $user = Arr::except($user, ['profile_photo', 'mobile_verified_at', 'is_social_account']);

            return response()->json([
                'message' => __('api.user.profile_update'),
                'data' => $user,
            ]);
        } catch (ValidationException $e) {
            return response()->json([
                'message' => $e->getMessage(),
            ], 500);
        } catch (\Exception $e) {
            return response()->json([
                'message' =>  __('api.user.update_error'),
            ], 500);
        }
    }

    public function show()
    {
        $user = auth()->user()->only(['id', 'name', 'email', 'isd_code', 'mobile', 'profile_photo', 'mobile_verified_at', 'stripe_id', 'sendbird_chat_id', 'is_social_account']);

        if ($user['profile_photo'] == null) {
            $user['profile_image'] = "";
        } else {
            $user['profile_image'] = $user['profile_photo'] == null ? '' : config('services.aws.aws_cloud_url') . $user['profile_photo'];
        }
        if ($user['mobile_verified_at'] == null) {
            $user['is_verified'] = 0;
        } else {
            $user['is_verified'] = 1;
        }
        if ($user['is_social_account'] == 0) {
            $user['login_type'] = 0;
            $user['social_id'] = "";
        } else {
            $social_details = SocialLogin::where('user_id', $user['id'])->first();
            $user['login_type'] = $social_details->type;
            $user['social_id'] = $social_details->social_id;
        }
        $user['rating'] = "0.0";
        $user['is_subscribed'] = 0;
        $user['is_profile_completed'] = 0;
        $user['step'] = 0;
        $user = Arr::except($user, ['profile_photo', 'mobile_verified_at', 'is_social_account']);
        return response()->json([
            'message' => __('rest-auth::label.ok'),
            'data' => [
                'user' => $user
            ]
        ]);
    }

    public function updateSocialUser(Request $request)
    {
        $user = User::findOrFail(auth()->user()->id);
        $user->update([
            'isd_code' => $request->isd_code,
            'mobile' => $request->mobile,
            'gender' => $request->gender,
            'dob' => $request->dob
        ]);
    }

    public function sendOTPOnMobile($request)
    {
        $otp = $this->generateOTP();
        $device_type = auth()->user()->device->device_type;
        // Send SMS on mobile number via HTTP client
        $receiverNumber = '"' . $request->isd_code . $request->mobile . '"';
        if ($device_type == "2") {
            $message = "Your one time password for 4u taxi booking app is : " . $otp . ". fjmjoET9YaI";
        } else {
            $message = "Your one time password for 4u taxi booking app is " . $otp;
        }

        try {

            $account_sid = config('services.twilio.account_sid');
            $auth_token = config('services.twilio.auth_token');
            $twilio_number = config('services.twilio.twilio_number');

            $client = new Client($account_sid, $auth_token);
            $client->messages->create($receiverNumber, [
                'from' => $twilio_number,
                'body' => $message
            ]);

            // dd('SMS Sent Successfully.');

        } catch (Exception $e) {
            // dd("Error: ". $e->getMessage());
        }
        return $otp;
    }

    public function deleteAccount()
    {
        try {
            // Begin a database transaction
            DB::beginTransaction();

            // Get the currently authenticated user
            $user = auth()->user();
           
            // Soft delete the user account
            $user->delete();

            // Commit the database transaction
            DB::commit();

            // Return a success response indicating that the account has been deleted
            return response()->success(__('api.deleteAccount'));

            // Return an error response if the user is not found
            return response()->error(__('api.not_found'), 404);
        } catch (Exception $e) {
            // Roll back the database transaction in case of an exception and return an error response with the exception message
            DB::rollBack();
            return response()->error($e->getMessage(), 500);
        }
    }
}
