<?php

namespace App\Http\Controllers\APIs\v1;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\UserAddress;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Exception;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;

class UserAddressController extends Controller
{
    public function get_list(Request $request)
    {
        $addresses = UserAddress::select(
            'id',
            'apartment_name',
            'street_address',
            'city',
            'postal_code',
            'add_lat',
            'add_long',
            'address_type',
            DB::raw('CASE 
        WHEN address_type = 1 THEN "Home"
        WHEN address_type = 2 THEN "Office"
        WHEN address_type = 3 THEN "Other"
        ELSE ""
    END AS address_text')
        )->where('user_id', auth()->user()->id)->orderBy('id', 'desc')->get();
        return response()->json([
            'message' => __('api.user_address.get_all_address'),
            'data' => $addresses
        ]);
    }


    public function add_address(Request $request)
    {
        try {
            $request->validate([
                'apartment_name' => 'required|string',
                'street_address' => 'required|string',
                'city' => 'required|string|max:100',
                'postal_code' => 'required|string|max:10',
                'address_type' => 'required|integer|in:1,2,3',
                'add_lat' => 'required',
                'add_long' => 'required'
            ], [
                'apartment_name.required' => 'Please enter a valid apartment or unit number.',
                'street_address.required' => 'Please enter a valid street address.',
                'city.required' => 'Please enter a valid city name.',
                'postal_code.required' => 'Please enter a valid postal code (ZIP code).'
            ]);

            UserAddress::create([
                'user_id' => auth()->user()->id,
                'apartment_name' => $request->apartment_name,
                'street_address' => $request->street_address,
                'city' => $request->city,
                'postal_code' => $request->postal_code,
                'address_type' => $request->address_type,
                'add_lat' => $request->add_lat,
                'add_long' => $request->add_long,
            ]);

            return response()->json([
                'message' => __('api.user_address.address_added_success'),
            ]);
        } catch (ValidationException $e) {
            return response()->json([
                'message' => $e->getMessage(),
            ], 422); // 400 Bad Request status code for validation failure
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                // 'error' => $e->getMessage(),
            ], 500); // 500 Internal Server Error status code for other exceptions
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $request->validate([
                'apartment_name' => 'required|string',
                'street_address' => 'required|string',
                'city' => 'required|string|max:100',
                'postal_code' => 'required|string|max:10',
                'address_type' => 'required|integer|in:1,2,3',
                'add_lat' => 'required',
                'add_long' => 'required'
            ], [
                'apartment_name.required' => 'Please enter a valid apartment or unit number.',
                'street_address.required' => 'Please enter a valid street address.',
                'city.required' => 'Please enter a valid city name.',
                'postal_code.required' => 'Please enter a valid postal code (ZIP code).'
            ]);
            $address = UserAddress::find($id);

            if (!$address) {
                return response()->json([
                    'message' => __('api.user_address.address_not_found'),
                ], 404);
            }

            $address->update($request->all());

            return response()->json([
                'message' => __('api.user_address.address_updated_success'),
            ]);
        } catch (ValidationException $e) {
            return response()->json([
                'message' => $e->getMessage(),
            ], 422); // 400 Bad Request status code for validation failure
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                // 'error' => $e->getMessage(),
            ], 500); // 500 Internal Server Error status code for other exceptions
        }
    }

    public function destroy($id)
    {
        try {
            $userAddress = UserAddress::findOrFail($id);
            $userAddress->delete();

            return response()->json([
                'message' => __('api.user_address.address_deleted_success'),
            ]);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'message' => __('api.user_address.no_data_found'),
            ]);
        } catch (Exception $e) {
            return response()->json([
                'message' => __('api.user_address.error_occurred'),
            ]);
        }
    }

    public function get_address_with_type(Request $request)
    {

        try {
            $address = UserAddress::select(
                'id',
                'apartment_name',
                'street_address',
                'city',
                'postal_code',
                'add_lat',
                'add_long',
                'address_type',
                DB::raw('CASE 
            WHEN address_type = 1 THEN "home"
            WHEN address_type = 2 THEN "office"
            WHEN address_type = 3 THEN "other"
            ELSE ""
        END AS address_text')
            )
                ->where('address_type', $request->type)->where('deleted_at', null)->get();

            return response()->json([
                'message' => __('api.user_address.address_fetch_success'),
                'data' => $address
            ]);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'message' => __('api.user_address.no_data_found'),
            ]);
        } catch (Exception $e) {
            return response()->json([
                'message' => __('api.user_address.error_occurred'),
            ]);
        }
    }
}
