<?php

namespace App\Http\Controllers\APIs\v1;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Twilio\Rest\Client;
use Twilio\TwiML\VoiceResponse;
use App\Models\MaskAssignment;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;


class TwilioController extends Controller
{
    private $twilioClient;
    public function __construct()
    {
        $accountSid = config('services.twilio.account_sid');
        $authToken = config('services.twilio.auth_token');
        $this->twilioClient = new Client($accountSid, $authToken);
    }

    public function assignMaskNumber(Request $request)
    {
        $incomingPhoneNumbers = $this->twilioClient->incomingPhoneNumbers->read();

        foreach ($incomingPhoneNumbers as $phoneNumber) {
            $twilioNumber = $phoneNumber->phoneNumber;

            if (!MaskAssignment::where('mask_number', $twilioNumber)->where('status', 1)->exists()) {
                break;
            }
        }

        if(empty($twilioNumber)) {
            return response()->json([
                'message' => __('api.twilio_number_assigned_failed'),
            ]);
        }

        $assignMaskNumber = MaskAssignment::create([
            'mask_number' => $twilioNumber,
            'status' => 1, // Status is initially set to 1
            'from_user_id' => Auth::user()->id,
            'to_user_id' => $request->user_id,
        ]);

        if ($assignMaskNumber) {
            return response()->json([
                'message' => __('api.twilio_number_assigned_success'),
                'mask_number' => $twilioNumber
            ]);
        } 
        return response()->json([
            'message' => __('api.twilio_lines_busy'),
        ]);
    }

    public function initiateCall(Request $request)
    {
        Log::info('initiateCall request received:', ['request' => $request->all()]);
        try {
            //$calledNumber = $request->input('To');
            $callerNumber = $request->input('From'); //Twilio mask number

            $calledNumber = MaskAssignment::query()
                ->selectRaw('CONCAT(users.isd_code, users.mobile) AS vMobileNumber')
                ->join('users', 'users.id', '=', 'mask_assignments.to_user_id')
                ->where('mask_assignments.mask_number', $callerNumber)
               // ->whereRaw('CONCAT(users.vISDCode, users.vMobileNumber) = ?', [$calledNumber])
                ->where('mask_assignments.status', 1)
                ->value('vMobileNumber');

            if (empty($calledNumber)) {
                Log::error(__('api.twilio_no_mask_number'));
                return response()->json([
                    'message' => __('api.twilio_no_mask_number'),
                ]);
            }

            // Initiate the call using the Twilio PHP library
            $this->twilioClient->account->calls->create(
                $calledNumber,
                $callerNumber,
                [
                    'url' => route('api.twilio.incoming.call'),
                    'statusCallback' => route('api.twilio.complete.call'),
                    'statusCallbackMethod' => 'POST',
                ]
            );
            Log::info(__('api.twilio_call_init_success'), ['calledNumber' => $calledNumber, 'callerNumber' => $callerNumber]);
        } catch (\Exception $e) {
            Log::error(__('api.twilio_call_init_failed'));
        }
    }


    public function handleIncomingCall(Request $request)
    {
        Log::info('Incoming call request received:', ['request' => $request->all()]);
        // Log the call status event
        $callStatus = $request->input('CallStatus');
        Log::info("Incoming Call Status: $callStatus");

        $calledNumber = $request->input('To');
        $callerNumber = $request->input('From'); //Twilio mask number

        Log::info("Incoming Call To: $calledNumber");

        $newStatus = $this->mapCallStatusToStatusCode($callStatus);

        $maskAssignment = MaskAssignment::join('users', 'users.id', '=', 'mask_assignments.to_user_id')
            ->where('mask_assignments.mask_number', $callerNumber)
            ->whereRaw('CONCAT(users.isd_code, users.mobile) = ?', [$calledNumber])
            ->where('mask_assignments.status', 1);

        if ($callStatus === 'in-progress') {
            $maskAssignment->update([
                'status' => $newStatus,
                'call_start_time' => now(),
            ]);
        } else {
            $maskAssignment->update(['status' => $newStatus]);
        }

        $twiml = new VoiceResponse();
        $twiml->say('Hello! Thank you for calling. This is a Twilio call example.');
        $twiml->hangup();

        return response($twiml)->header('Content-Type', 'text/xml');
    }

    public function handleCallCompletion(Request $request)
    {
        $callStatus = $request->input('CallStatus');
        Log::info("Call Completion Status: $callStatus");

        $callSid = $request->input('CallSid');
        $callDuration = $request->input('CallDuration');
        $calledNumber = $request->input('To');
        $callerNumber = $request->input('From'); //Twilio mask number

        Log::info("Call SID: $callSid");
        Log::info("Call Duration: $callDuration seconds");
        Log::info("Caller's Phone Number: $callerNumber");

        $newStatus = $this->mapCallStatusToStatusCode($callStatus);

        // Check for a timeout and update the status
        if ($request->input('Timeout') === 'true') {
            $newStatus = 0;
        }

        $maskAssignment = MaskAssignment::join('users', 'users.id', '=', 'mask_assignments.to_user_id')
            ->where('mask_assignments.mask_number', $callerNumber)
            ->whereRaw('CONCAT(users.isd_code, users.mobile) = ?', [$calledNumber])
            ->where('mask_assignments.status', 1);

        if ($callStatus === 'completed') {
            $maskAssignment->update([
                'status' => $newStatus,
                'call_end_time' => now(),
            ]);
        } else {
            $maskAssignment->update(['status' => $newStatus]);
        }

        // Handle different call statuses
        $twiml = new VoiceResponse();

        switch ($callStatus) {
            case 'completed':
                $twiml->say('Thank you for using our service. Goodbye.');
                break;

            case 'busy':
                $twiml->say('The call recipient is busy. Goodbye.');
                break;

            case 'failed':
                $twiml->say('The call has failed. Goodbye.');
                break;

            case 'no-answer':
                $twiml->say('There was no answer. Goodbye.');
                break;

            default:
                $twiml->say('Call status processed.');
                break;
        }

        return response($twiml)->header('Content-Type', 'text/xml');
    }


    private function mapCallStatusToStatusCode($callStatus)
    {
        // Map call statuses to status codes
        $statusMapping = [
            'completed' => 0,
            'assigned' => 1,
            'in-progress' => 2,
            'ringing' => 3,
            'busy' => 4,
            'failed' => 5,
            'no-answer' => 6,
        ];

        return $statusMapping[$callStatus] ?? 1;
    }

}
