<?php

namespace App\Http\Controllers\APIs\v1;

use App\Http\Controllers\Controller;
use App\Http\Requests\AddBankRequest;
use App\Http\Requests\DefaultBankRequest;
use App\Http\Requests\DeleteBankRequest;
use Illuminate\Http\Request;
use App\Http\Requests\UpdateDriverLicenseRequest;
use App\Http\Requests\UpdateDriverRequest;
use App\Http\Requests\UpdateVehicleInfoRequest;
use App\Http\Requests\UpdateVehicleImagesRequest;
use App\Http\Resources\BankResource;
use App\Models\BankAccount;
use App\Models\Driver;
use App\Models\User;
use App\Models\UserDevice;
use App\Models\UserSubscription;
use App\Models\VehicleInfo;
use App\Services\StripeService;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Stripe\Stripe;
use Exception;
use Illuminate\Support\Arr;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;
use SpaceO\RESTAuth\Traits\SendOTPTrait;
use Twilio\Rest\Client;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Redis;
use Stripe\Account;
use Stripe\Token;

class DriverController extends Controller
{

    use SendOTPTrait;
    private int $otpLength;
    protected $stripeService;


    public function __construct(StripeService $stripeService)
    {
        Stripe::setApiKey(config('services.stripe.secret'));
        $this->otpLength = config('rest_auth.otp_length');
        $this->stripeService = $stripeService;
    }

    public function updateDriverLicense(UpdateDriverLicenseRequest $request)
    {
        try {
            $driver = $this->findDriver(auth()->user()->id);
            $driver->update([
                'license_front_image' => $request->filled('license_front_image') ? $request->license_front_image : $driver->license_front_image,
                'license_back_image' => $request->filled('license_back_image') ? $request->license_back_image : $driver->license_back_image
            ]);
            $driver->step = 2;
            $driver->is_approved = 0;
            $driver->online_status = 0;
            $driver->save();

            return response()->json([
                'message' => __('api.driver.license_updated'),
            ]);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'message' => __('api.driver.not_found'),
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'message' => __('api.driver.license_error'),
            ], 500);
        }
    }

    public function updateVehicleInfo(UpdateVehicleInfoRequest $request)
    {
        try {
            $driverId = auth()->user()->id;
            $step = 3;
            $this->updateDriverInfo($step, $driverId, $request);

            return response()->json([
                'message' => __('api.driver.vehicle_update'),
            ]);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'message' => __('api.driver.not_found'),
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'message' => __('api.driver.vehicle_error'),
            ], 500);
        }
    }

    public function updateVehicleImages(UpdateVehicleImagesRequest $request)
    {
        try {
            $driverId = auth()->user()->id;
            $step = 4;
            $this->updateDriverInfo($step, $driverId, $request);

            return response()->json([
                'message' => __('api.driver.image_update'),
            ]);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'message' => __('api.driver.not_found'),
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'message' => __('api.driver.image_error'),
            ], 500);
        }
    }

    public function updateOnlineStatus(Request $request)
    {
        try {
            $driver = $this->findDriver(auth()->user()->id);
            if ($driver->is_approved != 1) {
                return response()->json([
                    'message' => __('api.driver.not_approved'),
                ], 403);
            }
            if ($driver->online_status == 1)
                $driver->online_status = 0;
            else
                $driver->online_status = 1;
            $driver->save();
            return response()->json([
                'message' => __('api.driver.status_update'),
            ]);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'message' => __('api.driver.not_found'),
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'message' => __('api.driver.status_error'),
            ], 500);
        }
    }

    public function getProfile(Request $request)
    {
        $user = auth()->user()->only(['id', 'full_name', 'email', 'isd_code', 'mobile', 'mobile_verified_at', 'profile_image', 'stripe_id', 'sendbird_chat_id', 'is_profile_completed', 'step']);

        $user['name'] = $user['full_name'];
        $user['login_type'] = 0;
        $user['social_id'] = "";
        $checksubscription = UserSubscription::where('subscribable_type', 'App\Models\Driver')->where('subscribable_id', $user['id'])->first();
        if ($checksubscription) {
            $user['is_subscribed'] = 1;
        } else {
            $user['is_subscribed'] = 0;
        }
        if ($user['mobile_verified_at'] == null) {
            $user['is_verified'] = 0;
        } else {
            $user['is_verified'] = 1;
        }
        $user['rating'] = "0.0";
        $user = Arr::except($user, ['full_name', 'mobile_verified_at']);
        $user['profile_image'] = $user['profile_image'] == null ? '' : config('services.aws.aws_cloud_url') . $user['profile_image'];
        return response()->json([
            'message' => __('rest-auth::label.ok'),
            'data' => [
                'user' => $user
            ]
        ]);
    }

    public function updateProfile(UpdateDriverRequest $request)
    {
        try {
            $driver = $this->findDriver(auth()->user()->id);
            if (isset($request->email)) {

                $driver->email = $request->email;
            }
            if (isset($request->full_name)) {
                $driver->full_name = $request->full_name;
            }

            if (isset($request->profile_photo)) {
                if (config('services.aws.aws_cloud_url') . $driver->profile_image != $request->profile_photo) {
                    $driver->profile_image = $request->profile_photo;
                }
            }
            if (isset($request->isd_code)) {
                $driver->isd_code = $request->isd_code;
            }
            if (isset($request->mobile) && ($driver->mobile != $request->mobile)) {
                $driver->mobile = $request->mobile;
                $driver->mobile_verified_at = null;
                $otp = $this->sendOTPOnMobile($request);
                Driver::where('id', $driver['id'])->update(
                    array_combine($this->OTPFields(), [
                        $otp, now()->addMinutes(config('rest_auth.otp_expires_in'))
                    ])
                );
            }
            $driver->save();
            $data = $driver->only(['id', 'full_name', 'email', 'isd_code', 'mobile', 'profile_image', 'mobile_verified_at', 'stripe_id', 'sendbird_chat_id', 'is_profile_completed', 'step']);
            $data['name'] = $data['full_name'];
            if ($data['mobile_verified_at'] == null) {
                $data['is_verified'] = 0;
            } else {
                $data['is_verified'] = 1;
            }
            $checksubscription = UserSubscription::where('subscribable_type', 'App\Models\Driver')->where('subscribable_id', $data['id'])->first();
            if ($checksubscription) {
                $data['is_subscribed'] = 1;
            } else {
                $data['is_subscribed'] = 0;
            }
            $data['login_type'] = 0;
            $data['social_id'] = "";
            $data['rating'] = "0.0";

            $data = Arr::except($data, ['full_name', 'mobile_verified_at']);
            $data['profile_image'] = $data['profile_image'] == null ? '' : config('services.aws.aws_cloud_url') . $data['profile_image'];
            return response()->json([
                'message' => __('api.driver.profile_update'),
                'data' => $data,
            ]);
        } catch (ValidationException $e) {
            return response()->json([
                'message' => $e->getMessage(),
            ], 500);
        } catch (Exception $e) {
            return response()->json([
                'message' => __('api.driver.update_error'),
            ], 500);
        }
    }

    protected function findDriver($driverId)
    {
        return Driver::findOrFail($driverId);
    }

    protected function updateDriverInfo($step, $driverId, $request)
    {
        // VehicleInfo::updateOrCreate(['driver_id' => $driverId], $data);

        $driver = $this->findDriver($driverId);
        $driver->vehicleinfo()->update([
            'vehicle_type' => $request->filled('vehicle_type') ? $request->vehicle_type : $driver->vehicleinfo->vehicle_type,
            'plate_number' => $request->filled('plate_number') ? $request->plate_number : $driver->vehicleinfo->plate_number,
            'model_name' => $request->filled('model_name') ? $request->model_name : $driver->vehicleinfo->model_name,
            'model_no' =>  $request->filled('model_no') ? $request->model_no : $driver->vehicleinfo->model_no,
            'color' => $request->filled('color') ? $request->color : $driver->vehicleinfo->color,
            'identification_no' => $request->filled('identification_no') ? $request->identification_no : $driver->vehicleinfo->identification_no,
            'model_year' =>  $request->filled('model_year') ? $request->model_year : $driver->vehicleinfo->model_year,
            'seating_capacity' => $request->filled('seating_capacity') ? $request->seating_capacity : $driver->vehicleinfo->seating_capacity,
            'registration_proof_file' => $request->filled('registration_proof_file') ? $request->registration_proof_file : $driver->vehicleinfo->registration_proof_file,
            'insurance_proof_file' => $request->filled('insurance_proof_file') ? $request->insurance_proof_file : $driver->vehicleinfo->insurance_proof_file,
            'vehicle_front_image' => $request->filled('vehicle_front_image') ? $request->vehicle_front_image : $driver->vehicleinfo->vehicle_front_image,
            'vehicle_back_image' => $request->filled('vehicle_back_image') ? $request->vehicle_back_image : $driver->vehicleinfo->vehicle_back_image,
            'vehicle_right_image' => $request->filled('vehicle_right_image') ? $request->vehicle_right_image : $driver->vehicleinfo->vehicle_right_image,
            'vehicle_left_image' => $request->filled('vehicle_left_image') ? $request->vehicle_left_image : $driver->vehicleinfo->vehicle_left_image,
        ]);
        $driver->step = $step;
        $driver->is_approved = 0;
        $driver->online_status = 0;
        $driver->save();
    }

    public function addBankAccount(AddBankRequest $request)
    {
        try {
            $driver = auth()->user();
            $customerId = $driver->stripe_id;
            $driverId = $driver->id;
            $is_default = false;

            if (!BankAccount::where('driver_id', $driverId)->where('is_default', 1)->exists()) {
                $is_default = true;
            }

            $token = Token::create([
                'bank_account' => [
                    'country' => 'US',
                    'currency' => 'usd',
                    'account_holder_name' => $request->acc_holder_name,
                    'account_holder_type' => $request->acc_type,
                    'routing_number' => $request->routing_number,
                    'account_number' => $request->acc_number,
                ],
            ]);

            $stripeCustomer = Account::createExternalAccount(
                $customerId,
                [
                    'external_account' => $token->id,
                ]
                // Add any additional customer data as needed
            );
            BankAccount::create([
                'driver_id' => $driverId,
                'bank_token' => $stripeCustomer->id,
                'is_default' => ($is_default == true) ? 1 : 0,
            ]);
            $driver = $this->findDriver($driverId);
            if ($driver->step == 4 && $driver->is_profile_completed == 0) {
                $driver->step = 5;
                $driver->is_profile_completed = 1;
                $driver->save();
            }

            return response()->json([
                'message' => __('api.driver.add_bank_acc'),

            ]);
        } catch (Exception $e) {
            // Handle any errors that occur during the card creation process
            return response()->json([
                'message' => __('api.driver.err_add_bank'),
                'error' => $e->getMessage(),
            ], 500);
        };
    }

    public function getDriverID(Request $request)
    {
        try {
            $get_details = Driver::select('license_front_image', 'license_back_image')->where('id', auth()->user()->id)->first();
            $get_details['license_front_image'] = ($get_details->license_front_image == "") ? "" : config('services.aws.aws_cloud_url') . $get_details->license_front_image;
            $get_details['license_back_image'] = ($get_details->license_back_image == "") ? "" : config('services.aws.aws_cloud_url') . $get_details->license_back_image;
            return response()->json([
                'message' => __('api.driver.license_fetch'),
                'data' => $get_details

            ]);
        } catch (Exception $e) {
            // Handle any errors that occur during the card creation process
            return response()->json([
                'message' => __('api.driver.commonerr'),
                // 'error' => $e->getMessage(),
            ], 500);
        };
    }

    public function getVehicleImages(Request $request)
    {
        try {
            $get_details = VehicleInfo::select('registration_proof_file', 'insurance_proof_file', 'vehicle_front_image', 'vehicle_back_image', 'vehicle_right_image', 'vehicle_left_image')->where('driver_id', auth()->user()->id)->first();
            $get_details['registration_proof_file'] = ($get_details->registration_proof_file == "") ? "" : config('services.aws.aws_cloud_url') . $get_details->registration_proof_file;
            $get_details['insurance_proof_file'] = ($get_details->insurance_proof_file == "") ? "" : config('services.aws.aws_cloud_url') . $get_details->insurance_proof_file;
            $get_details['vehicle_front_image'] = ($get_details->vehicle_front_image == "") ? "" : config('services.aws.aws_cloud_url') . $get_details->vehicle_front_image;
            $get_details['vehicle_back_image'] = ($get_details->vehicle_back_image == "") ? "" : config('services.aws.aws_cloud_url') . $get_details->vehicle_back_image;
            $get_details['vehicle_right_image'] = ($get_details->vehicle_right_image == "") ? "" : config('services.aws.aws_cloud_url') . $get_details->vehicle_right_image;
            $get_details['vehicle_left_image'] = ($get_details->vehicle_left_image == "") ? "" : config('services.aws.aws_cloud_url') . $get_details->vehicle_left_image;
            return response()->json([
                'message' => __('api.driver.vehicleimage_fetch'),
                'data' => $get_details

            ]);
        } catch (Exception $e) {
            // Handle any errors that occur during the card creation process
            return response()->json([
                'message' => __('api.driver.commonerr'),
                // 'error' => $e->getMessage(),
            ], 500);
        };
    }
    public function getVehicleInfo(Request $request)
    {
        try {
            $get_details = VehicleInfo::select('vehicle_type', 'plate_number', 'model_name', 'model_no', 'color', 'identification_no', 'model_year', 'seating_capacity')->where('driver_id', auth()->user()->id)->first();
            $get_details["vehicle_type"] = $get_details->vehicle_type ?? "";
            $get_details["plate_number"] = $get_details->plate_number ?? "";
            $get_details["model_name"] = $get_details->model_name ?? "";
            $get_details["model_no"] = $get_details->model_no ?? "";
            $get_details["color"] = $get_details->color ?? "";
            $get_details["identification_no"] = $get_details->identification_no ?? "";
            $get_details["model_year"] = $get_details->model_year ?? "";
            $get_details["seating_capacity"] = $get_details->seating_capacity ?? "";
            return response()->json([
                'message' => __('api.driver.vehicleinfo_fetch'),
                'data' => $get_details

            ]);
        } catch (Exception $e) {
            // Handle any errors that occur during the card creation process
            return response()->json([
                'message' => __('api.driver.commonerr'),
                // 'error' => $e->getMessage(),
            ], 500);
        };
    }

    public function getBankAccInfo(Request $request)
    {
        try {
            $get_details = BankAccount::select('acc_holder_name', 'routing_number', 'acc_number', 'acc_type')->where('driver_id', auth()->user()->id)->first();
            return response()->json([
                'message' => __('api.driver.bankinfo_fetch'),
                'data' => $get_details

            ]);
        } catch (Exception $e) {
            // Handle any errors that occur during the card creation process
            return response()->json([
                'message' => __('api.driver.commonerr'),
                // 'error' => $e->getMessage(),
            ], 500);
        };
    }

    public function sendOTPOnMobile($request)
    {
        $otp = $this->generateOTP();
        $device_type = auth()->user()->device->device_type;

        // Send SMS on mobile number via HTTP client
        $receiverNumber = '"' . $request->isd_code . $request->mobile . '"';
        if ($device_type == "2") {
            $message = "Your one time password for 4u taxi booking app is : " . $otp . ". D5g134hIW39";
        } else {
            $message = "Your one time password for 4u taxi booking app is " . $otp;
        }

        try {

            $account_sid = config('services.twilio.account_sid');
            $auth_token = config('services.twilio.auth_token');
            $twilio_number = config('services.twilio.twilio_number');

            $client = new Client($account_sid, $auth_token);
            $client->messages->create($receiverNumber, [
                'from' => $twilio_number,
                'body' => $message
            ]);

            // dd('SMS Sent Successfully.');

        } catch (Exception $e) {
            // dd("Error: ". $e->getMessage());
        }
        return $otp;
    }

    public function getNearbyDrivers(Request $request)
    {
        $userLat = $request->latitude;
        $userLong = $request->longitude;
        $maxDistance = 10;
        $drivers = Driver::select('latitude', 'longitude')
            ->selectRaw("{$maxDistance} * 2 * ASIN(SQRT(POWER(SIN(({$userLat} - latitude) * pi()/180 / 2), 2) + COS({$userLat} * pi()/180) * COS(latitude * pi()/180) * POWER(SIN(({$userLong} - longitude) * pi()/180 / 2), 2))) as distance")
            ->having('distance', '<', $maxDistance)
            ->orderBy('distance')
            ->get();
        $data['user_id'] = auth()->user()->id;
        $data['notification_count'] = 0;
        $data['quotes_count'] = 0;
        $data['drivers'] = $drivers;
        return response()->json([
            'message' => "list of nearby drivers",
            'data' => $data,
        ], 200);
    }

    public function getStatus(Request $request)
    {
        try {
            $driver = $this->findDriver(auth()->user()->id);
            return response()->json([
                'message' => __('api.driver.status_retrieve'),
                'data' => $driver->online_status
            ]);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'message' => __('api.driver.not_found'),
            ], 404);
        } catch (Exception $e) {
            return response()->json([
                'message' => __('api.driver.status_error'),
            ], 500);
        }
    }

    public function getBankList(Request $request)
    {
        try {
            $driver = auth()->user();
            $customerId = $driver->stripe_id;
            $bank_list = $this->stripeService->getBankList(
                $customerId,
            );
            $data = collect($bank_list);
            $db_bank = BankAccount::where('driver_id', $driver->id)->get();
            $data->each(function (&$item) use ($db_bank) {
                // Find the matching item in the second collection based on 'id' and 'card_token'
                $matchingItem = $db_bank->firstWhere('bank_token', $item['id']);

                if ($matchingItem) {
                    $item['db_id'] = $matchingItem['id'];
                    $item['is_default'] = $matchingItem['is_default'];
                }
            });




            return response()->json([
                'message' => __('api.bank.bank_list'),
                'data' => BankResource::collection($data),
            ]);
        } catch (\Exception $e) {
            // Handle any errors that occur during the card creation process
            return response()->json([
                'message' => __('api.bank.list_error'),
                'error' => $e->getMessage(),
            ], 500);
        };
    }

    public function deleteBank(DeleteBankRequest $request)
    {
        try {

            $customerId = auth()->user()->stripe_id;
            $bank = BankAccount::findOrFail($request->id);
            if ($bank->is_default == 0) {
                $stripeCustomer = $this->stripeService->deleteBank($customerId, $bank->bank_token);
                if ($stripeCustomer->deleted == true) {
                    $bank->delete();
                }

                return response()->json([
                    'message' => __('api.bank.delete_success'),

                ]);
            } else {
                return response()->json([
                    'message' => __('api.bank.delete_default_error'),

                ], 500);
            }
        } catch (\Exception $e) {
            // Handle any errors that occur during the card creation process
            return response()->json([
                'message' => __('api.bank.delete_error'),
                // 'error' => $e->getMessage(),
            ], 500);
        };
    }

    public function makeDefault(DefaultBankRequest $request)
    {
        try {
            $driver = auth()->user();
            $driverId = $driver->id;
            $customerId = $driver->stripe_id;
            $bank = BankAccount::findOrFail($request->id);
            if ($bank->is_default == 0) {
                $stripeCustomer = $this->stripeService->makeDefaultBank($customerId, $bank->bank_token);
                $ex_bank = $driver->banks()->where('is_default', 1)->update([
                    'is_default' => 0
                ]);
                $bank->is_default = 1;
                $bank->save();
                return response()->json([
                    'message' => __('api.bank.default_success'),

                ]);
            } else {
                return response()->json([
                    'message' => __('api.bank.already_default'),

                ], 500);
            }
        } catch (\Exception $e) {
            // Handle any errors that occur during the card creation process
            return response()->json([
                'message' => __('api.bank.default_error'),
                // 'error' => $e->getMessage(),
            ], 500);
        };
    }

    public function getHomeDetails()
    {
        $driver = auth()->user();
        $data =[
            "is_approved" => $driver->is_approved,
            "notification_count" => 0
        ];
        return response()->json([
            'message' => __('api.driver.home_details'),
            'data' => $data,
        ]);
    }

    public function deleteAccount()
    {
        try {
            // Begin a database transaction
            DB::beginTransaction();

            // Get the currently authenticated user
            $user = auth()->user();
           
            // Soft delete the user account
            $user->delete();

            // Commit the database transaction
            DB::commit();

            // Return a success response indicating that the account has been deleted
            return response()->success(__('api.deleteAccount'));

            // Return an error response if the user is not found
            return response()->error(__('api.not_found'), 404);
        } catch (Exception $e) {
            // Roll back the database transaction in case of an exception and return an error response with the exception message
            DB::rollBack();
            return response()->error($e->getMessage(), 500);
        }
    }
}
