<?php

namespace App\Http\Controllers\APIs\v1;

use App\Http\Controllers\Controller;
use App\Http\Requests\AddCardRequest;
use App\Http\Requests\DefaultCardRequest;
use App\Http\Requests\DeleteCardRequest;
use App\Http\Resources\CardResource;
use Illuminate\Http\Request;
use App\Services\StripeService;
use App\Models\Card;
use Illuminate\Support\Facades\DB;
use SebastianBergmann\CodeUnit\FunctionUnit;

class CardController extends Controller
{
    protected $stripeService;

    public function __construct(StripeService $stripeService)
    {
        $this->stripeService = $stripeService;
    }

    public function addCard(AddCardRequest $request)
    {
        try {
            $user = auth()->user();
            $customerId = $user->stripe_id;
            $is_default = false;
            if (isset($request->is_default) && $request->is_default == 1) {
                $user->cards()->where('is_default', 1)->update(
                    [
                        'is_default' => 0
                    ]
                );
            } else {
                if (!Card::where('user_id', $user->id)->exists()) {
                    $is_default = true;
                }
            }
            $this->stripeService->createCard($customerId, $request->card_token);
            Card::create([
                'user_id' => $user->id,
                'card_token' => $request->card_token,
                'is_default' => ($is_default == true) ? 1 : (isset($request->is_default) ? $request->is_default : 0),
            ]);
            return response()->json([
                'message' => __('api.card.add_success'),

            ]);
        } catch (\Exception $e) {
            // Handle any errors that occur during the card creation process
            return response()->json([
                'message' => __('api.card.add_error'),
                'error' => $e->getMessage(),
            ], 500);
        };
    }

    public function getCardList(Request $request)
    {
        try {
            $user = auth()->user();
            $customerId = $user->stripe_id;
            $card_list = $this->stripeService->getCardList(
                $customerId,
            );
            $data = collect($card_list);
            $db_card = Card::where('user_id', $user->id)->get();

            $data->each(function (&$item) use ($db_card) {
                // Find the matching item in the second collection based on 'id' and 'card_token'
                $matchingItem = $db_card->firstWhere('card_token', $item['id']);

                if ($matchingItem) {
                    $item['db_id'] = $matchingItem['id'];
                    $item['is_default'] = $matchingItem['is_default'];
                }
            });

            return response()->json([
                'message' => __('api.card.card_list'),
                'data' => CardResource::collection($data),
            ]);
        } catch (\Exception $e) {
            // Handle any errors that occur during the card creation process
            return response()->json([
                'message' => __('api.card.list_error'),
                'error' => $e->getMessage(),
            ], 500);
        };
    }

    public function deleteCard(DeleteCardRequest $request)
    {
        try {

            $customerId = auth()->user()->stripe_id;
            $card = Card::findOrFail($request->id);
            if ($card->is_default == 0) {
            $stripeCustomer = $this->stripeService->deleteCard($customerId, $card->card_token);
            
            if ($stripeCustomer->id == $card->card_token) {
                $card->delete();
            }

            return response()->json([
                'message' => __('api.card.delete_success'),

            ]);
        }
        else{
            return response()->json([
                'message' => __('api.card.default_delete_error'),

            ]);
        }
        } catch (\Exception $e) {
            // Handle any errors that occur during the card creation process
            return response()->json([
                'message' => __('api.card.delete_error'),
                // 'error' => $e->getMessage(),
            ], 500);
        };
    }

    public function makeDefault(DefaultCardRequest $request)
    {
        try {
            $user = auth()->user();
            $ex_card = $user->cards()->where('is_default', 1)->update(
                [
                    'is_default' => 0
                ]
            );
            $get_card = Card::where('id', $request->id)->first();
            $get_card->is_default = 1;
            $get_card->save();

            return response()->json([
                'message' => __('api.card.default_success'),

            ]);
        } catch (\Exception $e) {
            // Handle any errors that occur during the card creation process
            return response()->json([
                'message' => __('api.card.default_error'),
                // 'error' => $e->getMessage(),
            ], 500);
        };
    }
}
