<?php

namespace App\Http\Controllers\APIs\v1;

use App\Http\Controllers\Controller;
use App\Http\Requests\BookingRequest;
use App\Http\Requests\RideCancelRequest;
use App\Http\Requests\StartScheduleRide;
use App\Http\Requests\UpdateCurrentRideRequest;
use App\Http\Requests\UpdateLocationRequest;
use App\Http\Resources\DriverBookingStatusResource;
use App\Http\Resources\UserBookingStatusResource;
use App\Jobs\CheckBookingForBids;
use App\Jobs\ScheduleRideJob;
use App\Models\Booking;
use App\Models\BookingAddress;
use App\Models\BookingPayment;
use App\Models\CancelBooking;
use App\Models\Card;
use App\Models\Driver;
use App\Models\Promocode;
use App\Models\Review;
use App\Models\User;
use App\Models\UserDevice;
use App\Services\StripeService;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Redis;
use Illuminate\Validation\ValidationException;

class BookingController extends Controller
{
    protected $stripeService;

    public function __construct(StripeService $stripeService)
    {
        $this->stripeService = $stripeService;
    }
    public function rideBooking(BookingRequest $request)
    {
        try {
            $user = User::find(auth()->user()->id);

            $pickupLat = $request->pickup_lat;
            $pickupLong = $request->pickup_long;
            $radiusMeters = 100;
            $carId = $request->car_id;
            $driversWithinRadius = Driver::select('id', 'latitude', 'longitude', DB::raw("
            (3959 * acos(cos(radians($pickupLat)) * cos(radians(latitude)) *
            cos(radians(longitude) - radians($pickupLong)) +
            sin(radians($pickupLat)) * sin(radians(latitude))))
            AS distance
            "))
                ->having('distance', '<=', $radiusMeters)
                ->where('online_status', '=', 1)
                ->where('is_occupied', '=', 0)
                ->whereHas('vehicleinfo', function ($query) use ($carId) {
                    $query->where('vehicle_type', $carId);
                })
                ->orderBy('distance', 'asc')
                ->get();
            Log::info('driver within radius');
            Log::info($driversWithinRadius);
            $drivers = $driversWithinRadius->makeHidden(['latitude', 'longitude', 'distance']);
            // dd($drivers);
            $validatedData = $request->validated();

            $user = auth()->user();
            if (!$user) {
                throw new ValidationException(__('api.booking.unauthenticated'));
            }

            $bookingData = $this->prepareBookingData($validatedData, $user);
            $booking = Booking::create($bookingData);
            $this->createBookingAddress($booking, $request);
            $bookingId = $booking->id;

            if ($request->is_scheduled == 1) {
                $scheduledDateTime = $request->schedule_date . ' ' . $request->schedule_time;

                // Calculate the timestamp by subtracting 1 hour (3600 seconds)
                $timestamp = strtotime($scheduledDateTime) - 3600;
                $booking['last_time'] = $timestamp;
                ScheduleRideJob::dispatch($booking)->delay(now()->timestamp($timestamp));
            }

            // if(!empty($request->user_price))
            // {

            // }

            if ($bookingId && $request->is_scheduled == 0) {
                $carbonDate = Carbon::parse($booking->booking_date_time);

                // Add one minute
                $updatedDate = $carbonDate->addMinute();

                // Format the updated date
                $updatedDateFormatted = $updatedDate->format('Y-m-d H:i:s');
                $request_data = [
                    'bid_id' =>  0,
                    'booking_id' => $bookingId,
                    'pickup_location' => $booking->bookingaddress->pickup_location ?? '',
                    'dropoff_location' => $booking->bookingaddress->dropoff_location ?? '',
                    'miles' => 3.2,
                    'submitted_price' => (float)$request->user_price ?? 0.00,
                    'user_bid_amount' => 0.00,
                    'driver_bid_amount' => 0.00,
                    'user_image' => (auth()->user()->profile_photo == null) ? "" : config('services.aws.aws_cloud_url') . auth()->user()->profile_photo,
                    'user_name' => auth()->user()->name ?? '',
                    'user_vehicle' => $booking->car->car_type_name ?? '',
                    'date_time' => $booking->booking_date_time,
                    'bid_expire_time' => $updatedDateFormatted,
                    'is_scheduled' => (int)$booking->is_scheduled
                ];
                $driverarr = [];
                if (!empty($drivers)) {
                    foreach ($drivers as $driver) {
                        $driverarr[]['id'] = $driver->id;
                        sendPushNotification($user->id, $driver->id, 'New Booking Arrived', 'A new customer has requested a ride. Check the details and get ready to respond.', $booking->id, 1, 'user', 'driver', 2);
                    }
                }
                Log::info('driverarr');
                Log::info($driverarr);

                Log::info('bookingdata');
                Log::info($request_data);
                // dd($driverarr);
                $redis = Redis::connection();
                $redis->publish('get_customer_request', json_encode(["drivers" => $driverarr, "bookingdata" => $request_data]));
                CheckBookingForBids::dispatch($bookingId)->delay(now()->addMinutes(2));
            }

            Log::info('booking');
            Log::info($booking->id);
            return response()->json([
                'message' => __('api.booking.success'),
                'data' => $booking->id
            ]);
        } catch (ValidationException $e) {
            return $this->errorResponse(__('api.booking.validation_error'), $e->errors(), 422);
        } catch (\Exception $e) {
            return $e->getMessage();
            // return $this->errorResponse(__('api.booking.error'), $e->getMessage(), 500);
        }
    }

    public function applyPromocode(Request $request)
    {
        try {
            $today = Carbon::now()->format('Y-m-d');
            $promocode = Promocode::where('promocode_name', $request->promocode)
                ->where('start_date', '<=', $today)
                ->where('end_date', '>=', $today)
                ->where('is_active', 1)
                ->first();

            if (!$promocode) {
                throw new \Exception(__('api.booking.promocode_not_found'));
            }
            if ($promocode->discount_type == 1) {
                $msg = "Flat $" . $promocode->discount . "discount";
            } else {
                $msg = "Flat " . $promocode->discount . "% discount";
            }
            // $booking = $this->findBooking($request->booking_id);
            // $booking->promocode_id = $promocode->id;
            // $booking->save();
            $data = [
                'id' => $promocode->id,
                'promocode_msg' => $msg,
                'promocode' => $request->promocode
            ];
            return response()->json([
                'message' => __('api.booking.apply_promocode'),
                'data' => $data
            ]);
        } catch (\Exception $e) {
            return $this->errorResponse(__('api.booking.promocode_error'), $e->getMessage(), 500);
        }
    }

    // public function removePromocode(Request $request)
    // {
    //     try {
    //         $booking = $this->findBooking($request->booking_id);
    //         $booking->promocode_id = null;
    //         $booking->save();

    //         return response()->json([
    //             'message' => __('api.booking.remove_promocode')
    //         ]);
    //     } catch (\Exception $e) {
    //         return $this->errorResponse(__('api.booking.remove_promocode_error'), $e->getMessage(), 500);
    //     }
    // }

    public function rideCancel(RideCancelRequest $request)
    {
        try {
            $booking = $this->findBooking($request->booking_id);
            $booking->booking_status = 7;
            CancelBooking::create([
                'booking_id' => $request->booking_id,
                'cancel_reason_id' => $request->cancel_reason_id,
                'cancel_description' => $request->description,
                'cancelled_by' => $request->cancelled_by,
            ]);
            $booking->save();
            Driver::where('id', $booking->driver_id)->update(['is_occupied' => 0]);
            $rating = 0;
            $rating = Review::where('reviewable_id', $booking->driver->id)->where('reviews.reviewable_type', '=', 'App\Models\Driver')->avg('rating');
            $booking['rating'] = $rating;
            $user_request_data = new UserBookingStatusResource($booking);
            $driver_request_data = new DriverBookingStatusResource($booking);
            $redis = Redis::connection();
            if ($request->cancelled_by == 2) {
                $redis->publish('user_accept_bid', json_encode(["user" => $booking->user_id, "bookingdata" => $user_request_data]));
            } else {
                $redis->publish('driver_accept_bid', json_encode(["driver" => $booking->driver_id, "bookingdata" => $driver_request_data]));
            }

            return response()->json([
                'message' => __('api.booking.cancel')
            ]);
        } catch (\Exception $e) {
            return $this->errorResponse(__('api.booking.cancel_error'), $e->getMessage(), 500);
        }
    }

    public function getBookings(Request $request)
    {
        try {
            $userId = auth()->user()->id;
            $bookingStatus = null;
            $isScheduled = null;

            switch ($request->status) {
                case 1:
                    $bookingStatus = [1];
                    $isScheduled = [1];
                    break;
                case 2:
                    $bookingStatus = [6];
                    $isScheduled = [0, 1];
                    break;
                case 3:
                    $bookingStatus = [7];
                    $isScheduled = [0, 1];
                    break;
            }

            $bookings = Booking::select('id', 'user_id', 'booking_no', 'driver_id', 'booking_date_time as booking_date', 'booking_status', 'booking_amount', 'is_scheduled', 'schedule_time', 'schedule_date')
                ->with([
                    'driver:id,full_name,profile_image,sendbird_chat_id,mobile',
                    'driver.vehicleinfo:driver_id,plate_number',
                    'bookingaddress:booking_id,pickup_location,dropoff_location,pickup_lat,pickup_long,dropoff_lat,dropoff_long'
                ])
                ->where('user_id', $userId)
                ->whereIn('booking_status', $bookingStatus)
                ->whereIn('is_scheduled', $isScheduled)
                ->orderBy('id', 'desc')
                ->paginate($request->offset, ['*'], 'page', $request->page);

            $finalData = $bookings->map(function ($booking) {
                $driver = $booking->driver;
                $carbonTime = Carbon::parse($booking->schedule_time);
                $updatedTime = $carbonTime->subMinutes(10);
                $scheduled_time = $booking->schedule_date . " " . $booking->schedule_time;
                $start_button_time = $booking->schedule_date . " " . $updatedTime->toTimeString();
                return [
                    'id' => $booking->id,
                    'booking_no' => $booking->booking_no,
                    'pickup_location' => $booking->bookingaddress->pickup_location ?? "",
                    'dropoff_location' => $booking->bookingaddress->dropoff_location ?? "",
                    'booking_date_time' => $booking->booking_date,
                    'booking_amount' => (float)$booking->booking_amount,
                    'is_scheduled' => $booking->booking_status == 1 ? ($booking->driver_id == null ? 0 : 1) : $booking->is_scheduled,
                    'scheduled_time' => $booking->is_scheduled == 1 ? $scheduled_time : "",
                    'start_button_time' => $booking->is_scheduled == 1 ? $start_button_time : "",
                    'booking_status' => $booking->booking_status,
                    'driver_image' => $driver ? (($driver->profile_image == null) ? '' : config('services.aws.aws_cloud_url') . $driver->profile_image) : '',
                    'driver_name' => $driver ? $driver->full_name : '',
                    'driver_vehicle_no' => $driver ? ($driver->vehicleinfo->plate_number ?? '') : '',
                    'driver_chat_id' => ($booking->booking_status != 5 && $booking->booking_status != 9) ? ($driver ? $driver->sendbird_chat_id : "") : "",
                    'driver_phone_no' => ($booking->booking_status != 5 && $booking->booking_status != 9) ? ($driver ? $driver->mobile : "") : "",
                ];
            });

            return response()->json([
                'message' => __('api.booking.fetch_success'),
                'data' => $finalData,
                'pagination' => [
                    'current_page' => $bookings->currentPage(),
                    'per_page' => $bookings->perPage(),
                    'total' => $bookings->total(),
                    'total_page' => $bookings->lastPage()
                ],
            ]);
        } catch (\Exception $e) {
            return $this->errorResponse(__('api.booking.fetch_success_error'), $e->getMessage(), 500);
        }
    }

    public function getDriverBookings(Request $request)
    {
        try {
            $userId = auth()->user()->id;
            $bookingStatus = null;
            $isScheduled = null;

            switch ($request->status) {
                case 1:
                    $bookingStatus = [1];
                    $isScheduled = [1];
                    break;
                case 2:
                    $bookingStatus = [6];
                    $isScheduled = [0, 1];
                    break;
                case 3:
                    $bookingStatus = [7];
                    $isScheduled = [0, 1];
                    break;
            }

            $bookings = Booking::select('id', 'user_id', 'booking_no', 'driver_id', 'booking_date_time as booking_date', 'booking_status', 'booking_amount', 'schedule_time', 'is_scheduled', 'schedule_date')
                ->with([
                    'driver:id,full_name,profile_image,sendbird_chat_id,mobile',
                    'users:id,name,profile_photo,sendbird_chat_id,mobile',
                    'driver.vehicleinfo:driver_id,plate_number',
                    'bookingaddress:booking_id,pickup_location,dropoff_location,pickup_lat,pickup_long,dropoff_lat,dropoff_long'
                ])
                ->where('driver_id', $userId)
                ->whereIn('booking_status', $bookingStatus)
                ->whereIn('is_scheduled', $isScheduled)
                ->orderBy('id', 'desc')
                ->paginate($request->offset, ['*'], 'page', $request->page);

            $finalData = $bookings->map(function ($booking) {
                $user = $booking->users;
                $carbonTime = Carbon::parse($booking->schedule_time);
                $updatedTime = $carbonTime->subMinutes(10);
                $scheduled_time = $booking->schedule_date . " " . $booking->schedule_time;
                $start_button_time = $booking->schedule_date . " " . $updatedTime->toTimeString();
                return [
                    'id' => $booking->id,
                    'booking_no' => $booking->booking_no,
                    'pickup_location' => $booking->bookingaddress->pickup_location ?? "",
                    'dropoff_location' => $booking->bookingaddress->dropoff_location ?? "",
                    'booking_date_time' => $booking->booking_date,
                    'booking_amount' => (float)$booking->booking_amount,
                    'is_scheduled' => $booking->booking_status == 1 ? ($booking->driver_id == null ? 0 : 1) : $booking->is_scheduled,
                    'scheduled_time' => $booking->is_scheduled == 1 ? $scheduled_time : "",
                    'start_button_time' => $booking->is_scheduled == 1 ? $start_button_time : "",
                    'booking_status' => $booking->booking_status,
                    'user_image' => $user ? (($user->profile_photo == null) ? '' : config('services.aws.aws_cloud_url') . $user->profile_photo) : '',
                    'user_name' => $user ? $user->name : '',
                    'user_chat_id' => ($booking->booking_status != 5 && $booking->booking_status != 9) ? ($user ? $user->sendbird_chat_id : "") : "",
                    'user_phone_no' => ($booking->booking_status != 5 && $booking->booking_status != 9) ? ($user ? $user->mobile : "") : "",
                ];
            });

            return response()->json([
                'message' => __('api.booking.fetch_success'),
                'data' => $finalData,
                'pagination' => [
                    'current_page' => $bookings->currentPage(),
                    'per_page' => $bookings->perPage(),
                    'total' => $bookings->total(),
                    'total_page' => $bookings->lastPage()
                ],
            ]);
        } catch (\Exception $e) {
            return $this->errorResponse(__('api.booking.fetch_success_error'), $e->getMessage(), 500);
        }
    }

    public function getBookingDetails(Request $request)
    {
        try {
            $bookingDetails = Booking::with([
                'driver:id,profile_image,full_name,sendbird_chat_id,mobile',
                'driver.vehicleinfo:driver_id,plate_number',
                'bookingaddress:booking_id,pickup_location,dropoff_location,pickup_lat,pickup_long,dropoff_lat,dropoff_long',
                'cancelbooking:booking_id,cancel_reason_id,cancel_description,cancelled_by',
                'cancelbooking.cancelreason:id,reason',
            ])->findOrFail($request->id);
            // dd($bookingDetails);
            $miles = calculateDistanceInMiles(
                $bookingDetails->bookingaddress->pickup_lat,
                $bookingDetails->bookingaddress->pickup_long,
                $bookingDetails->bookingaddress->dropoff_lat,
                $bookingDetails->bookingaddress->dropoff_long
            );
            $rating = Review::where('reviewable_id', $bookingDetails->driver->id)->where('reviews.reviewable_type', '=', 'App\Models\Driver')->avg('rating');
            $bookingDetails['rating'] = "" . round($rating, 2);
            $carbonTime = Carbon::parse($bookingDetails->schedule_time);
            $updatedTime = $carbonTime->subMinutes(10);
            $scheduled_time = $bookingDetails->schedule_date . " " . $bookingDetails->schedule_time;
            $start_button_time = $bookingDetails->schedule_date . " " . $updatedTime->toTimeString();
            $driver_profile_image = $bookingDetails->driver->profile_image == null ? '' : config('services.aws.aws_cloud_url') . $bookingDetails->driver->profile_image;
            $user_profile_image = $bookingDetails->users->profile_photo == null ? '' : config('services.aws.aws_cloud_url') . $bookingDetails->users->profile_photo;
            $data = [
                'id' => $bookingDetails->id,
                'driver_id' => $bookingDetails->driver_id,
                'booking_no' => $bookingDetails->booking_no,
                'booking_date_time' => $bookingDetails->booking_date_time,
                'profile_image' => (auth()->user()->isDriver() ? $driver_profile_image : $user_profile_image) ?? "",
                'name' => (auth()->user()->isDriver() ? $bookingDetails->users->name : $bookingDetails->driver->full_name) ?? "",
                'driver_rating' => "" . round($bookingDetails->rating, 2),
                'sendbird_chat_id' => (auth()->user()->isDriver() ? $bookingDetails->users->sendbird_chat_id : $bookingDetails->driver->sendbird_chat_id) ?? "",
                'mobile_no' => (auth()->user()->isDriver() ? $bookingDetails->users->isd_code . $bookingDetails->users->mobile : $bookingDetails->driver->isd_code . $bookingDetails->driver->mobile) ?? "",
                'vehicle_no' => $bookingDetails->driver->vehicleinfo->plate_number ?? '',
                'vehicle_name' => $bookingDetails->car->car_type_name ?? '',
                'pickup_location' => $bookingDetails->bookingaddress->pickup_location ?? "",
                'dropoff_location' => $bookingDetails->bookingaddress->dropoff_location ?? "",
                'pickup_lat' => $bookingDetails->bookingaddress->pickup_lat ?? "",
                'pickup_long' => $bookingDetails->bookingaddress->pickup_long ?? "",
                'dropoff_lat' => $bookingDetails->bookingaddress->dropoff_lat ?? "",
                'dropoff_long' => $bookingDetails->bookingaddress->dropoff_long ?? "",
                'miles' => $miles,
                'rate_given' => Review::where('booking_id', $bookingDetails->id)->count(),
                'booking_rating' => Review::where('booking_id', $bookingDetails->id)->first()->rating ?? 0,
                'booking_amount' => (float)($bookingDetails->booking_amount + $bookingDetails->driver_tip + $bookingDetails->waiting_charge) ?? 0.00,
                'is_scheduled' => $bookingDetails->booking_status == 1 ? ($bookingDetails->driver_id == null ? 0 : 1) : $bookingDetails->is_scheduled,
                'scheduled_time' => $bookingDetails->is_scheduled == 1 ? $scheduled_time : "",
                'start_button_time' => $bookingDetails->is_scheduled == 1 ? $start_button_time : "",
                'booking_status' => $bookingDetails->booking_status,
                'booking_cancelled_by' => $bookingDetails->booking_status == 5 ? ($bookingDetails->cancelbooking->cancelled_by == 1 ? "Customer" : "Driver") : "",
                'booking_cancel_reason' => $bookingDetails->booking_status == 5 ? $bookingDetails->cancelbooking->cancelreason->reason : "",
                'booking_tip' => (float)$bookingDetails->driver_tip ?? 0.00,
                'booking_waiting_charge' => (float)$bookingDetails->waiting_charge ?? 0.00,
                'driver_fare' => (float)$bookingDetails->booking_amount ?? 0.00
            ];

            return response()->json([
                'message' => __('api.booking.fetch_detail'),
                'data' => $data
            ]);
        } catch (\Exception $e) {
            return $this->errorResponse(__('api.booking.fetch_success'), $e->getMessage(), 500);
        }
    }

    public function updateCurrentRideStatus(UpdateCurrentRideRequest $request)
    {
        $booking = Booking::find($request->booking_id);
        if ($request->status == 2 && $booking->is_scheduled == 1) {
            if ($booking->driver->online_status == 1) {
                $booking->booking_status = $request->status;
                $booking->save();
                $rating = 0;
                $rating = Review::where('reviewable_id', $booking->driver->id)->where('reviews.reviewable_type', '=', 'App\Models\Driver')->avg('rating');
                $booking['rating'] = $rating;
                $user_request_data = new UserBookingStatusResource($booking);
                $driver_request_data = new DriverBookingStatusResource($booking);
                $redis = Redis::connection();
                $redis->publish('user_accept_bid', json_encode(["user" => $booking->user_id, "bookingdata" => $user_request_data]));
                $redis->publish('driver_accept_bid', json_encode(["driver" => $booking->driver_id, "bookingdata" => $driver_request_data]));
            } else {
                return response()->json([
                    'message' => __('api.booking.driver_status_update'),
                ]);
            }
        } else {
            $setting_waiting_min = 5;
            $setting_waiting_charge = 5;
            $currentTime = Carbon::now();

            $booking->booking_status = $request->status;
            if ($request->status == 3) {
                $booking->arrival_time = $currentTime;
            }
            if ($request->status == 4) {
                $arrival_time = Carbon::parse($booking->arrival_time);
                if ($arrival_time->diffInMinutes($currentTime) > $setting_waiting_min) {
                    $booking->waiting_charge = $setting_waiting_charge;
                }
            }
            if ($request->status == 6) {
                Driver::where('id', $booking->driver_id)->update(['is_occupied' => 0]);
                $gettoken = Card::where('id', $booking->payment_card_id)->first();
                $token = $gettoken->card_token;
                $promocode_amount = 0;
                if ($booking->promocode_id) {
                    $promocode = Promocode::find($booking->promocode_id);
                    if ($promocode->discount_type == 1) {
                        $promocode_amount = $booking->booking_amount - $promocode->discount;
                        $booking->promocode_amount = $promocode->discount;
                    } else {
                        $promocode_amount = $booking->booking_amount - (($booking->booking_amount * $promocode->discount) / 100);
                        $booking->promocode_amount = ($booking->booking_amount * $promocode->discount) / 100;
                    }
                }


                $final_amount = $booking->booking_amount + $booking->waiting_charge + $promocode_amount;
                $payment = $this->stripeService->createPaymentIntent($final_amount, $token, $booking->driver->stripe_id, $booking->users->stripe_id, "Ride Booking Amount");
                BookingPayment::create([
                    'booking_id' => $request->booking_id,
                    'payment_amount' => $final_amount,
                    'payment_method' => $payment['payment_method_types'][0],
                    'transaction_id' => $payment['id'],
                    'payment_status' => 2,
                    'payment_type' => 2
                ]);
            }
            $booking->save();
            $rating = 0;
            $rating = Review::where('reviewable_id', $booking->driver->id)->where('reviews.reviewable_type', '=', 'App\Models\Driver')->avg('rating');
            $booking['rating'] = $rating;
            $user_request_data = new UserBookingStatusResource($booking);
            $driver_request_data = new DriverBookingStatusResource($booking);
            $redis = Redis::connection();
            $redis->publish('user_accept_bid', json_encode(["user" => $booking->user_id, "bookingdata" => $user_request_data]));
            $redis->publish('driver_accept_bid', json_encode(["driver" => $booking->driver_id, "bookingdata" => $driver_request_data]));
        }


        return response()->json([
            'message' => __('api.booking.status_update'),
        ]);
    }

    private function prepareBookingData(array $validatedData, $user)
    {
        $user_price =  !empty($validatedData['user_price']) ?  $validatedData['user_price'] : 0;
        return [
            'user_id' => $user->id,
            'booking_no' => Carbon::now()->timestamp,
            'booking_date_time' => Carbon::now()->toDateTimeString(),
            'booking_status' => 1,
            'booking_amount' => 0,
            'user_price' => $user_price,
            'car_id' => $validatedData['car_id'],
            'payment_card_id' => $validatedData['payment_card_id'],
            'is_scheduled' => $validatedData['is_scheduled'],
            'schedule_date' => $validatedData['schedule_date'] ?? null,
            'schedule_time' => $validatedData['schedule_time'] ?? null,
        ];
    }

    public function getCurrentBookingDetails(Request $request)
    {
        try {
            $request_data = (object)[];
            $curr_user_id = auth()->user()->id;
            $is_occupied = 0;
            if (auth()->user()->isDriver()) {
                $is_occupied = auth()->user()->is_occupied;
                $booking = Booking::whereNotIn('booking_status', [1, 6, 7])
                    ->where('driver_id', $curr_user_id)
                    ->whereDate('booking_date_time', '>=', now()->toDateString())
                    ->orderBy('id', 'desc')
                    ->first();

                if ($booking) {
                    $rating = 0;
                    $rating = Review::where('reviewable_id', $booking->driver->id)->where('reviews.reviewable_type', '=', 'App\Models\Driver')->avg('rating');
                    $booking['rating'] = $rating;
                    $request_data = new DriverBookingStatusResource($booking);
                }
            } else {
                $booking = Booking::whereNotIn('booking_status', [1, 6, 7])
                    ->where('user_id', $curr_user_id)
                    ->whereDate('booking_date_time', '>=', now()->toDateString())
                    ->orderBy('id', 'desc')
                    ->first();
                if ($booking) {
                    $rating = 0;
                    $rating = Review::where('reviewable_id', $booking->driver->id)->where('reviews.reviewable_type', '=', 'App\Models\Driver')->avg('rating');
                    $booking['rating'] = $rating;
                    $request_data = new UserBookingStatusResource($booking);
                }
            }


            return response()->json([
                'message' => __('api.booking.current_ride'),
                'data' => $request_data,
                'is_occupied' => $is_occupied
            ]);
        } catch (\Exception $e) {
            return $this->errorResponse(__('api.booking.fetch_success'), $e->getMessage(), 500);
        }
    }

    private function createBookingAddress(Booking $booking, Request $request)
    {
        BookingAddress::create([
            'booking_id' => $booking->id,
            'pickup_location' => $request->pickup_location,
            'dropoff_location' => $request->dropoff_location,
            'pickup_lat' => $request->pickup_lat,
            'pickup_long' => $request->pickup_long,
            'dropoff_lat' => $request->dropoff_lat,
            'dropoff_long' => $request->dropoff_long,
            'promocode_id' => $request->promocode_id,
        ]);
    }

    private function findBooking(int $bookingId)
    {
        $booking = Booking::find($bookingId);
        if (!$booking) {
            throw new \Exception(__('api.booking.booking_not_found'));
        }
        return $booking;
    }

    private function errorResponse(string $message, $errors, int $statusCode)
    {
        return response()->json([
            'message' => $errors,
            // 'errors' => $errors
        ], $statusCode);
    }

    public function getCustomerRequest(Request $request)
    {
        try {
            $driver = auth()->user();
            $driver_lat = $driver->latitude;
            $driver_long = $driver->longitude;
            $oneMinuteAgo = now()->subMinute()->format('Y-m-d H:i:s');
            $oneMinuteLater = now()->addMinute()->format('Y-m-d H:i:s');
            $radiusMeters = 100;
            $bookings = DB::table('bookings')
                ->select('bookings.id as booking_id', 'bookings.user_price', 'bookings.is_scheduled', 'bookings.booking_date_time', 'bids.id as bid_id', 'users.name', 'users.profile_photo', 'bids.user_bid_amount', 'bids.driver_bid_amount', 'booking_addresses.*', 'cars.car_type_name', 'bids.updated_at as bid_updated_at')
                ->join('users', 'bookings.user_id', '=', 'users.id')
                ->leftJoin('bids', 'bookings.id', '=', 'bids.booking_id')
                ->leftJoin('cars', 'bookings.car_id', '=', 'cars.id')
                ->leftJoin('booking_addresses', 'bookings.id', '=', 'booking_addresses.booking_id')
                ->addSelect(DB::raw('
                (3959 * acos(cos(radians(booking_addresses.pickup_lat)) * cos(radians(' . $driver_lat . '))
                * cos(radians(' . $driver_long . ') - radians(booking_addresses.pickup_long))
                + sin(radians(booking_addresses.pickup_lat)) * sin(radians(' . $driver_lat . '))))
                AS distance_meters'))
                ->having('distance_meters', '<=', $radiusMeters)
                ->where('bookings.booking_status', config('constant.booking.pending'))
                ->where('bookings.driver_id', NULL)
                ->whereDate('bookings.booking_date_time', today())
                ->where(function ($query) use ($oneMinuteAgo, $oneMinuteLater, $driver) {
                    $query->whereNull('bids.id')
                        ->orWhere(function ($query) use ($oneMinuteAgo, $oneMinuteLater, $driver) {
                            $query->where(function ($query) use ($driver) {
                                $query->where('bids.driver_id', $driver->id)->whereIn('bids.bid_status', [0, 3]);
                            })
                                ->orWhere(function ($query) use ($oneMinuteAgo, $oneMinuteLater, $driver) {
                                    $query->where('bids.bid_status', 2)->where('bids.driver_id', $driver->id)
                                        ->whereBetween('bids.updated_at', [$oneMinuteAgo, $oneMinuteLater]);
                                });
                        });
                })
                ->where(function ($query) use ($oneMinuteAgo, $oneMinuteLater) {
                    $query->where('bookings.user_price', 0)
                        ->orWhere(function ($query) use ($oneMinuteAgo, $oneMinuteLater) {
                            $query->whereBetween('bookings.booking_date_time', [$oneMinuteAgo, $oneMinuteLater]);
                        });
                })
                ->orderBy('bookings.id', 'desc')
                ->get();

            $finalData = $bookings->map(function ($booking) {
                if ($booking->bid_id > 0) {
                    $carbonDate = Carbon::parse($booking->bid_updated_at);
                } else {
                    $carbonDate = Carbon::parse($booking->booking_date_time);
                }


                // Add one minute
                $updatedDate = $carbonDate->addMinute();

                // Format the updated date
                $updatedDateFormatted = $updatedDate->format('Y-m-d H:i:s');
                return [
                    'bid_id' =>  $booking->bid_id ? $booking->bid_id : 0,
                    'booking_id' => $booking->booking_id,
                    'pickup_location' => $booking->pickup_location ?? '',
                    'dropoff_location' => $booking->dropoff_location ?? '',
                    'miles' => 3.2,
                    'submitted_price' => (float)$booking->user_price ?? 0.00,
                    'user_bid_amount' => $booking->user_bid_amount ? (float)$booking->user_bid_amount : 0.00,
                    'driver_bid_amount' => $booking->driver_bid_amount ? (float)$booking->driver_bid_amount : 0.00,
                    'user_image' => ($booking->profile_photo == null) ? "" : config('services.aws.aws_cloud_url') . $booking->profile_photo,
                    'user_name' => $booking->name ?? '',
                    'user_vehicle' => $booking->car_type_name ?? '',
                    'date_time' => $booking->booking_date_time,
                    'bid_expire_time' => $updatedDateFormatted,
                    'is_scheduled' => $booking->is_scheduled
                ];
            });

            return response()->json([
                'message' => __('api.bids.bids_list'),
                'data' => $finalData
            ]);
        } catch (\Exception $e) {
            // Log the error if needed
            // Log::error($e->getMessage());

            return response()->json([
                'message' => $e->getMessage(),
            ], 500);
        }
    }

    public function updateLocation(UpdateLocationRequest $request)
    {

        try {
            $user_id = 0;
            if (auth()->user()->isDriver()) {
                $booking = Booking::whereNotIn('booking_status', [1, 6, 7])
                    ->where('driver_id', auth()->user()->id)
                    ->whereDate('booking_date_time', '>=', now()->toDateString())
                    ->orderBy('id', 'desc')
                    ->first();
                if ($booking) {
                    $user_id = $booking->user_id;
                }
                Driver::where('id', auth()->user()->id)->update([
                    'latitude' => $request->latitude,
                    'longitude' => $request->longitude
                ]);
                $data = [
                    'latitude' => $request->latitude,
                    'longitude' => $request->longitude,
                    'status' => 1
                ];
            } else {
                $booking = Booking::whereNotIn('booking_status', [1, 6, 7])
                    ->where('user_id', auth()->user()->id)
                    ->whereDate('booking_date_time', '>=', now()->toDateString())
                    ->orderBy('id', 'desc')
                    ->first();
                if ($booking) {
                    $user_id = $booking->driver_id;
                }
                User::where('id', auth()->user()->id)->update([
                    'latitude' => $request->latitude,
                    'longitude' => $request->longitude
                ]);
                $data = [
                    'latitude' => $request->latitude,
                    'longitude' => $request->longitude,
                    'status' => 1
                ];
            }

            // $redis = Redis::connection();
            // $redis->publish('location_update_data', json_encode(['message' => "hello"]));
            return response()->json([
                'message' => __('api.location_update.update_success'),
                'user' => $user_id,
                'data' => $data
            ], 200);
        } catch (Exception $e) {
            // Handle any exceptions that might occur during the process
            DB::rollback();
            return response()->json([
                'error' => __('api.location_update.update_error'),
            ], 500);
        }
    }

    public function checkBookingForBids()
    {
        $msg = "Searching for drivers";
        $booking = Booking::where('user_id', auth()->user()->id)
            ->whereDate('created_at', Carbon::today())
            ->orderBy('id', 'desc')->first();
        if ($booking) {
            $data = [
                'booking_id' => $booking->id ?? 0,
                'status' => true,
                'msg' => $msg
            ];
        } else {
            $data = [
                'booking_id' => $booking->id ?? 0,
                'status' => false,
                'msg' => $msg
            ];
        }
        return response()->json([
            'data' => $data
        ]);
    }
}
