<?php

namespace App\Http\Controllers\APIs\v1;

use App\Http\Controllers\Controller;
use App\Http\Requests\BidStatusUpdateRequest;
use App\Http\Resources\DriverBidResource;
use App\Http\Resources\DriverBookingStatusResource;
use App\Http\Resources\UserBidResource;
use App\Http\Resources\UserBookingStatusResource;
use App\Models\Bid;
use App\Models\Booking;
use App\Models\Driver;
use App\Models\Review;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Support\Facades\Redis;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;

class BidController extends Controller
{
    public function getBids(Request $request)
    {
        try {
            $userId = auth()->user()->id;
            $user = User::find(auth()->user()->id);
            $userLat = $user->latitude;
            $userLong = $user->longitude;
            $bid_status_arr = config('constant.bid');
            $bids = DB::table('bookings')
                ->join('bids', 'bookings.id', '=', 'bids.booking_id')
                ->join('booking_addresses', 'bookings.id', '=', 'booking_addresses.booking_id')
                ->join('drivers', 'bids.driver_id', '=', 'drivers.id')
                ->leftJoin('vehicle_infos', 'drivers.id', '=', 'vehicle_infos.driver_id')
                ->leftJoin('cars', 'vehicle_infos.vehicle_type', '=', 'cars.id') // Join with cars table using vehicle_type reference ID
                ->leftJoin('reviews', function ($join) {
                    $join->on('drivers.id', '=', 'reviews.reviewable_id')
                        ->where('reviews.reviewable_type', '=', 'App\Models\Driver');
                })
                ->select(
                    'bookings.id as booking_id',
                    'bids.id as bid_id',
                    'booking_addresses.pickup_location',
                    'booking_addresses.dropoff_location',
                    'booking_addresses.pickup_lat',
                    'booking_addresses.pickup_long',
                    'bookings.booking_date_time',
                    'bookings.user_price',
                    'bookings.is_scheduled',
                    'bids.driver_id',
                    'bids.driver_bid_amount',
                    'bids.user_bid_amount',
                    'drivers.profile_image as driver_image',
                    'drivers.full_name as driver_name',
                    DB::raw('cars.car_type_name as driver_vehicle'),
                    DB::raw('AVG(reviews.rating) as driver_rating'),

                )
                ->where('bookings.user_id', $userId)
                ->where('bids.bid_status', '!=', $bid_status_arr['reject'])
                ->where('bids.bid_status', '!=', $bid_status_arr['accept'])
                ->groupBy('bids.id', 'booking_addresses.pickup_location', 'booking_addresses.dropoff_location', 'booking_addresses.pickup_lat', 'booking_addresses.pickup_long', 'bids.driver_bid_amount', 'drivers.profile_image', 'drivers.full_name', 'cars.car_type_name')
                ->orderBy('bids.id', 'desc')
                ->get();

            $finalData = $bids->map(function ($bid) use ($userLat, $userLong) {
                $carbonDate = Carbon::parse($bid->booking_date_time);

                // Add one minute
                $updatedDate = $carbonDate->addMinute();

                // Format the updated date
                $updatedDateFormatted = $updatedDate->format('Y-m-d H:i:s');
                return [
                    'bid_id' => $bid->bid_id,
                    'booking_id' => $bid->booking_id,
                    'pickup_location' => $bid->pickup_location,
                    'dropoff_location' => $bid->dropoff_location,
                    'miles' => calculateDistanceInMiles($userLat, $userLong, $bid->pickup_lat, $bid->pickup_long),
                    'submitted_price' => (float)$bid->user_price ?? 0.00,
                    'user_bid_amount' => (float)$bid->user_bid_amount ?? 0.00,
                    'driver_bid_amount' => (float)$bid->driver_bid_amount ?? 0.00,
                    'driver_id' => $bid->driver_id,
                    'driver_image' => ($bid->driver_image == null) ? "" : config('services.aws.aws_cloud_url') . $bid->driver_image,
                    'driver_name' => $bid->driver_name ?? '',
                    'driver_vehicle' => $bid->driver_vehicle ?? '',
                    'driver_rating' => $bid->driver_rating ?? '',
                    'date_time' => $bid->booking_date_time,
                    'bid_expire_time' => $updatedDateFormatted,
                    'is_scheduled' => $bid->is_scheduled
                ];
            });

            return response()->json([
                'message' => __('api.bids.bids_list'),
                'data' => $finalData
            ]);
        } catch (\Exception $e) {
            // Log the error if needed
            // Log::error($e->getMessage());

            return response()->json([
                'message' => $e->getMessage(),
            ], 500);
        }
    }

    public function updateBidStatus(BidStatusUpdateRequest $request)
    {
        try {
            $msg = '';
            $booking_status_arr = config('constant.booking');
            $curr_user_id = auth()->user()->id;
            $bookingId = $request->booking_id;
            if ($request->driver_id == 0) {
                $request->merge(['driver_id' => $curr_user_id]);
            }
            $driverId = $request->driver_id;
            $bid = Bid::where('booking_id', $request->booking_id)->where('driver_id', $request->driver_id)->first();
            $booking = Booking::find($request->booking_id);
            $redis = Redis::connection();
            if ($request->status == 2) {
                $msg = __('api.bids.bid_counter');
                $bid->user_bid_amount = $request->bid_price;
                $bid->bid_status = $request->status;
                $bid->save();
                $rating = Review::where('reviewable_id', $booking->bid->driver->id)->where('reviews.reviewable_type', '=', 'App\Models\Driver')->avg('rating');
                $booking['rating'] = $rating;
                $booking['bid'] = $bid;
                $user_request_data = new UserBidResource($booking);
                $driver_request_data = new DriverBidResource($booking);
                sendPushNotification($booking->user_id, $bid->driver_id, 'Counter Bid Received by User', 'Passenger sent a counter. Review their proposal and choose to accept or make a new bid.', $bid->id, 6, 'user', 'driver', 2);
                $redis->publish('get_bid_request', json_encode(["user" => $booking->user_id, "bookingdata" => $user_request_data]));
                $redis->publish('get_driver_bid_request', json_encode(["driver" => $bid->driver_id, "bookingdata" => $driver_request_data]));
            } elseif ($request->status == 3) {
                $msg = "";
                if ($booking) {
                    if ($booking->is_scheduled == 0) {
                        $start_time = $booking->booking_date_time;
                    } else {
                        $start_time = $booking->schedule_fire_time;
                    }

                    $end_time = now();
                    if ($end_time->diffInMinutes($start_time) < 1) {
                        $msg = __('api.bids.bid_counter');
                        $bid = $this->createNewBid($request);
                        sendPushNotification(auth()->user()->id, $booking->user_id, 'Counter Bid Received by Driver', 'Driver has sent a counter. Review and decide to accept or make a new offer.', $bid->id, 7, 'driver', 'user', 1);
                        $rating = Review::where('reviewable_id', $booking->bid->driver->id)->where('reviews.reviewable_type', '=', 'App\Models\Driver')->avg('rating');
                        $booking['rating'] = $rating;
                        $booking['bid'] = $bid;
                        $user_request_data = new UserBidResource($booking);
                        $driver_request_data = new DriverBidResource($booking);
                        $redis->publish('get_driver_bid_request', json_encode(["driver" => $bid->driver_id, "bookingdata" => $driver_request_data]));
                        $redis->publish('get_bid_request', json_encode(["user" => $booking->user_id, "bookingdata" => $user_request_data]));
                    } else {
                        $msg = __('api.bids.bid_expire');
                        return response()->json([
                            'message' => $msg,
                        ], 500);
                    }
                } else {
                    $msg = __('api.bids.booking_not_available');
                    return response()->json([
                        'message' => $msg,
                    ], 500);
                }
            } elseif ($request->status == 4) {
                $msg = __('api.bids.bid_reject');
                if ($bid) {
                    $bid->bid_status = $request->status;
                    $bid->save();
                    $booking = Booking::find($request->booking_id);
                } else {
                    $booking = Booking::find($request->booking_id);
                    $bid = $this->createNewBid($request);
                }

                if (auth()->user()->isDriver()) {
                    sendPushNotification($curr_user_id, $booking->user_id, 'Bid Rejected by Driver', 'Driver declined your booking request. Wait for another driver', $bid->id, 4, 'driver', 'user', 1);
                } else {
                    sendPushNotification($booking->user_id, $curr_user_id, 'Bid Rejected by User', 'Passenger declined your bid. Consider revising your offer or waiting for next trip.', $bid->id, 5, 'user', 'driver', 2);
                }
            } elseif ($request->status == 1) {
                $curr_ride = Booking::where(function ($q) use ($driverId, $bookingId) {
                    $q->where(function ($q) use ($driverId) {
                        $q->where('driver_id', $driverId)->whereNotIn('booking_status', [1, 6, 7]);
                    })->orWhere(function ($q) use ($bookingId) {
                        $q->where('id', $bookingId)->whereNotNull("driver_id");
                    });
                })

                    ->whereDate('booking_date_time', '>=', now()->toDateString())
                    ->orderBy('id', 'desc')
                    ->exists();
                //  print_r($curr_ride);
                //  exit;
                if (!$curr_ride) {
                    if ($bid) {

                        $bid->bid_status = $request->status;
                        if (auth()->user()->isDriver() && $request->bid_price == null) {
                            $booking_price = $bid->user_bid_amount;
                        } else {
                            $booking_price = $bid->driver_bid_amount;
                        }
                        if ($booking->is_scheduled == 0) {
                            $update_booking = $bid->booking()->update([
                                'driver_id' => $bid->driver_id,
                                'booking_amount' => $booking_price,
                                'booking_status' => $booking_status_arr['on_the_way']
                            ]);
                        } else {
                            $up_booking = Booking::where('id', $request->booking_id)->update([
                                'driver_id' => $bid->driver_id,
                                'booking_amount' => $booking_price,
                            ]);
                        }
                        $update_driver = Driver::where('id', $driverId)->update(['is_occupied' => 1]);
                        $bid->save();
                        $msg = __('api.bids.bid_accept');
                    } else {

                        $booking_price = $booking->user_price;
                        if ($booking->is_scheduled == 0) {
                            $up_booking = Booking::where('id', $request->booking_id)->update([
                                'driver_id' => $curr_user_id,
                                'booking_amount' => $booking_price,
                                'booking_status' => $booking_status_arr['on_the_way']
                            ]);
                        } else {
                            $up_booking = Booking::where('id', $request->booking_id)->update([
                                'driver_id' => $curr_user_id,
                                'booking_amount' => $booking_price,
                            ]);
                        }
                        $update_driver = Driver::where('id', $driverId)->update(['is_occupied' => 1]);
                        $bid = $this->createNewBid($request);
                        $msg = __('api.bids.bid_accept');
                    }
                    if ($booking->is_scheduled == 0) {
                        $booking = Booking::find($request->booking_id);
                        $rating = 0;
                        $rating = Review::where('reviewable_id', $booking->driver->id)->where('reviews.reviewable_type', '=', 'App\Models\Driver')->avg('rating');
                        $booking['rating'] = $rating;
                        $user_request_data = new UserBookingStatusResource($booking);
                        $driver_request_data = new DriverBookingStatusResource($booking);
                        $redis->publish('user_accept_bid', json_encode(["user" => $booking->user_id, "bookingdata" => $user_request_data]));
                        $redis->publish('driver_accept_bid', json_encode(["driver" => $booking->driver_id, "bookingdata" => $driver_request_data]));
                        if (auth()->user()->isDriver()) {
                            sendPushNotification($curr_user_id, $booking->user_id, 'Bid Accepted by Driver', 'A driver has accepted your booking request. Your ride is confirmed.', $bid->id, 3, 'driver', 'user', 1);
                        } else {
                            sendPushNotification($curr_user_id, $booking->driver_id, 'Bid Accepted by User', 'Customer accepted your bid. Prepare for the scheduled pickup.', $bid->id, 2, 'user', 'driver', 2);
                        }
                    }
                } else {
                    if (auth()->user()->isDriver()) {
                        $msg = __('api.bids.booking_not_available');
                        return response()->json([
                            'message' => $msg,
                        ], 500);
                    } else {
                        $msg = __('api.bids.driver_not_available');
                        return response()->json([
                            'message' => $msg,
                        ], 500);
                    }
                }
            } else {
                $curr_ride = Booking::where(function ($q) use ($driverId, $bookingId) {
                    $q->where(function ($q) use ($driverId) {
                        $q->where('driver_id', $driverId)->whereNotIn('booking_status', [1, 6, 7]);
                    })->orWhere(function ($q) use ($bookingId) {
                        $q->where('id', $bookingId)->whereNotNull("driver_id");
                    });
                })

                    ->whereDate('booking_date_time', '>=', now()->toDateString())
                    ->orderBy('id', 'desc')
                    ->exists();
                if (!$curr_ride) {
                    $bid = $this->createNewBid($request);
                    sendPushNotification($curr_user_id, $booking->user_id, 'New Bid Received', 'Driver has sent a bid. Review and decide to accept or make a new offer."', $bid->id, 10, 'driver', 'user', 1);
                    $rating = Review::where('reviewable_id', $booking->bid->driver->id)->where('reviews.reviewable_type', '=', 'App\Models\Driver')->avg('rating');
                    $booking['rating'] = $rating;
                    $booking['bid'] = $bid;
                    $user_request_data = new UserBidResource($booking);
                    $driver_request_data = new DriverBidResource($booking);
                    $redis->publish('get_bid_request', json_encode(["user" => $booking->user_id, "bookingdata" => $user_request_data]));
                    $redis->publish('get_driver_bid_request', json_encode(["driver" => $bid->driver_id, "bookingdata" => $driver_request_data]));
                    $msg = __('api.bids.bid_success');
                } else {
                    $msg = __('api.bids.booking_not_available');
                    return response()->json([
                        'message' => $msg,
                    ], 500);
                }
            }

            return response()->json([
                'message' => $msg,
            ]);
        } catch (ValidationException $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => $e->errors(),
            ], 422);
        }
    }

    private function createNewBid($request)
    {
        $booking = Booking::select('user_price')->where('id', $request->booking_id)->first();
        $bid = new Bid();
        $bid->driver_id = auth()->user()->id;
        $bid->booking_id = $request->booking_id;
        $bid->user_bid_amount = $booking->user_price ?? null;
        $bid->driver_bid_amount = $request->bid_price ?? null;
        $bid->bid_status = $request->status;
        $bid->save();
        return $bid;
    }
}
