<?php

namespace App\Http\Controllers\APIs\v1;

use App\Http\Controllers\Controller;
use App\Http\Requests\BookAirTransportRequest;
use App\Http\Requests\UdateQuoteRequest;
use App\Models\AirTransportation;
use App\Models\AirTransportationBooking;
use App\Models\AirTransportBookingRequest;
use App\Models\CancelBooking;
use App\Models\Notification;
use App\Models\Quote;
use App\Models\Review;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;

class AirTransportationBookingController extends Controller
{
    public function bookAirTransport(BookAirTransportRequest $request)
    {
        try {
            DB::beginTransaction();
            $bookingData = $this->prepareBookingData($request);

           $booking = AirTransportationBooking::create($bookingData);
           $this->sendAirTransportationBookingRequest($booking);
           DB::commit();
            return response()->json([
                'message' => __('api.airtransport.booking_success'),
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::info($e);

            Log::info("message: " . $e->getMessage() . "line: " . $e->getLine());
            return response()->json([
                'message' => __('api.airtransport.booking_error'),
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    private function prepareBookingData(Request $request): array
    {
        $bookingData = [
            'user_id' => auth()->user()->id,
            'booking_no' => Carbon::now()->timestamp,
            'trip_type' => $request->trip_type,
            'pickup_location' => $request->pickup_location,
            'dropoff_location' => $request->dropoff_location,
            'departure_date_time' => $request->departure_date_time,
            'flying_type' => $request->flying_type,
            'comment' => $request->comment,
            'adult' => $request->adult,
            'child' => $request->child,
            'infants' => $request->infants,
            'status' => 1,
        ];

        if ($request->trip_type == 2) {
            $bookingData['arrival_date_time'] = $request->arrival_date_time;
        }

        return $bookingData;
    }

    private function sendAirTransportationBookingRequest($booking){
        $time = Carbon::parse($booking->departure_date_time)->format('H:i:s');
        $airTransportation =  AirTransportation::select([
                'air_transportations.id',
                'air_transportations.email',
                'air_transportations.first_name',
                'days.days',
                'pt.pickup_location',
                'dt.dropoff_location',
                'ot.start_time as override_start_time',
                'ot.end_time as override_end_time',
                'time_slots.start_time',
                'time_slots.end_time',
        ])
        ->leftJoin('days', function ($join) use($booking){
            $day = Carbon::parse($booking->departure_date_time)->format('l');
            $join->on('days.air_transportation_id', '=', 'air_transportations.id');
            $join->where('days.days', '=', strtolower($day));
        })
        ->rightJoin('terminals AS pt', function ($join) use($booking){
            $join->on('pt.air_transportation_id', '=','air_transportations.id');
            $join->where('pt.pickup_location', '=', DB::raw("'{$booking->pickup_location}'"));
        })
        ->rightJoin('terminals AS dt', function ($join)  use($booking){
            $join->on('dt.air_transportation_id', '=','air_transportations.id');
            $join->where('dt.dropoff_location', '=', $booking->dropoff_location);
        })
        ->leftJoin('time_slots', 'days.id', '=', 'time_slots.day_id')
        ->leftJoin('override_timeslots AS ot', function ($join) use($booking){
            $date = Carbon::parse($booking->departure_date_time)->format('Y-m-d');
            $join->on('ot.air_transportation_id', '=','air_transportations.id')
                ->where('ot.date', '=', $date);
        })
        ->whereNotNull('time_slots.id')
        ->where(DB::raw("IF(ot.id IS NOT NULL, ot.start_time, time_slots.start_time)") ,'<=', $time)
        ->where(DB::raw("IF(ot.id IS NOT NULL, ot.end_time, time_slots.end_time)"),'>=', $time)
        ->get();

        $bookingRequests = [];
        $mailData = [
            'name' => Auth::user()->name,
            'pickup_location' => $booking->pickup_location,
            'dropoff_location' => $booking->dropoff_location,
            'departure_date_time' => $booking->departure_date_time,
        ];
        foreach ($airTransportation as $request) {
            $bookingRequests[] = [
                'air_transportation_id' => $request->id,
                'air_booking_id' => $booking->id,
                'created_at' => now(),
                'updated_at' => now(),
            ];
            Mail::send('email.air-booking-request', $mailData, function ($message) use ($request) {
                $message->to($request->email)
                ->subject('Air Booking Request');
            });
        }

       
        // $emails = $airTransportation->pluck('email')->toArray();
       
        AirTransportBookingRequest::insert($bookingRequests);

    }

    public function cancelAirTransportBooking(Request $request)
    {
        try {
            $booking = AirTransportationBooking::findOrFail($request->booking_id);

            $cancel = new CancelBooking;

            $cancel->air_transportation_booking_id = $booking->id;
            $cancel->cancel_reason_id = $request->reason_id;
            $cancel->cancel_description = $request->reason;
            $cancel->cancelled_by = 1;
            $cancel->type = 3;
            $cancel->save();

            $booking->cancel_reason = $request->reason;
            $booking->status = 4;
            $booking->save();

            return response()->json([
                'message' => __('api.airtransport.booking_cancel'),
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => __('api.airtransport.cancel_fail'),
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    public function getQuotes(Request $request)
    {
        try {
            $userId = auth()->user()->id;
            $quotes = Quote::with([
                'air_transportation_bookings:id,booking_no,pickup_location,dropoff_location',
                'air_transportations:id,profile_image,first_name,last_name,company_name,sendbird_chat_id,mobile',
            ])
                ->whereHas('air_transportation_bookings', function ($query) use ($userId) {
                    $query->where('user_id', $userId)->where('quote_status', 2);
                })
                ->orderBy('id','desc')
                ->paginate($request->offset, ['*'], 'page', $request->page);

            $finalData = $quotes->map(function ($quote) {

                return [
                    'id' => $quote->id,
                    'booking_no' => '#'.$quote->air_transportation_bookings->booking_no,
                    'pickup_location' => $quote->air_transportation_bookings->pickup_location,
                    'dropoff_location' => $quote->air_transportation_bookings->dropoff_location,
                    'quote_date_time' => formatDate($quote->quote_date_time),
                    'company_quote_price' => $quote->company_quote_price,
                    'air_transportation_image' => $quote->air_transportations->profile_image ?? '',
                    'air_transportation_name' => $quote->air_transportations->getFullName(),
                    'air_transportation_chat_id' => $quote->air_transportations->sendbird_chat_id,
                    'air_transportation_phone_no' => $quote->air_transportations->mobile,
                ];
            });

            return response()->json([
                'message' => __('api.quote.quote_list'),
                'data' => $finalData,
                'pagination' => [
                    'current_page' => $quotes->currentPage(),
                    'per_page' => $quotes->perPage(),
                    'total' => $quotes->total(),
                    'total_page' => $quotes->lastPage()
                ],
            ]);
        } catch (\Exception $e) {
            // Log the error if needed
            // Log::error($e->getMessage());

            return response()->json([
                'message' => __('api.quote.quote_error'),
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function getQuotesDetails(Request $request)
    {
        try {
            $quoteId = $request->id;

            $quoteDetails = Quote::select('id', 'air_transportation_booking_id', 'air_transportation_id', 'quote_date_time', 'company_quote_price', 'user_quote_price', 'quote_status')
                ->with('air_transportation_bookings', 'air_transportations:id,profile_image,first_name,last_name,company_name,sendbird_chat_id,mobile')
                ->findOrFail($quoteId);

            $booking = $quoteDetails->air_transportation_bookings;
            $airTransportation = $quoteDetails->air_transportations;
            if ($quoteDetails->user_quote_price == null && $quoteDetails->quote_status == 2) {
                $quote_status = 0;
            } else {
                $quote_status = $quoteDetails->quote_status;
            }
            $data = [
                'id' => $quoteDetails->id,
                'booking_no' => '#'.$booking->booking_no,
                'quote_date_time' => formatDate($quoteDetails->quote_date_time),
                'air_transportation_image' => $airTransportation->profile_image ?? '',
                'air_transportation_name' => $airTransportation->getFullName(),
                'air_transportation_chat_id' => $airTransportation->sendbird_chat_id,
                'air_transportation_phone_no' => $airTransportation->mobile,
                'adult' => $booking->adult,
                'child' => $booking->child,
                'flying_type' => getFlyingTypeText($booking->flying_type),
                'trip_type' => $booking->trip_type,
                'pickup_location' => $booking->pickup_location,
                'dropoff_location' => $booking->dropoff_location,
                'departure_date_time' => quoteDate($booking->departure_date_time),
                'arrival_date_time' => $booking->arrival_date_time == null ? '' : quoteDate($booking->arrival_date_time),
                'quotation' => $quoteDetails->company_quote_price,
                'user_quote_price' => $quoteDetails->user_quote_price ?? "",
                'quote_status' => $quote_status,
                'comment' => $booking->comment,
            ];

            return response()->json([
                'message' => __('api.quote.quote_success'),
                'data' => $data
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => __('api.quote.quote_error'),
            ], 404);
        }
    }

    public function updateQuoteStatus(UdateQuoteRequest $request)
    {
        try {
            $quote = Quote::findOrFail($request->id);
            $msg = "";

            switch ($request->status) {
                case 1:
                    $booking = AirTransportationBooking::findOrFail($quote->air_transportation_booking_id);
                    $booking->air_transportation_id = $quote->air_transportation_id;
                    $booking->status = 2;
                    if ($request->quote_price == "") {
                        $booking->booking_price = $quote->company_quote_price;
                    } else {
                        $booking->booking_price = $request->quote_price;
                    }
                    $booking->save();
                    $quote->quote_status = 1;
                    $msg = __('api.quote.quote_accept');
                    // sendPushNotification($booking->user_id, 'Quote accepted', 'quote accept by company', $request->id, '');
                    break;
                case 2:
                    $quote->user_quote_price = $request->quote_price;
                    $quote->quote_status = 2;
                    $msg = __('api.quote.quote_counter');
                    break;
                case 3:
                    $quote->quote_status = 3;
                    $msg = __('api.quote.quote_reject');
                    break;
                default:
                    return response()->json([
                        'message' => __('api.quote.invalid_status'),
                    ], 422);
            }
            $quote->save();
            return response()->json([
                'message' => $msg,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => __('api.quote.error'),
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function getBookings(Request $request)
    {
        try {
            $userId = auth()->user()->id;
            $status = $request->status;

            $bookingsQuery = AirTransportationBooking::select('id', 'air_transportation_id', 'booking_no', 'pickup_location', 'dropoff_location', 'created_at as booking_date', 'status', 'booking_price')
                ->where('user_id', $userId)
                ->where('status', $status);

            if ($status != 1) {
                $bookingsQuery->with('air_transportations:id,first_name,last_name,company_name,sendbird_chat_id,mobile');
            }

            $bookings = $bookingsQuery->orderBy('id','desc')->paginate($request->offset, ['*'], 'page', $request->page);

            $finalData = $bookings->map(function ($booking) {
                $airTransportation = $booking->air_transportations;

                return [
                    'id' => $booking->id,
                    'booking_no' => '#'.$booking->booking_no,
                    'pickup_location' => $booking->pickup_location,
                    'dropoff_location' => $booking->dropoff_location,
                    'booking_date_time' => formatDate($booking->booking_date),
                    'air_transportation_image' => $airTransportation ? ($airTransportation->profile_image == null ? '' : $airTransportation->profile_image) : '',
                    'air_transportation_name' => $airTransportation ? $airTransportation->getFullName() : 'Not Applicable',
                    'air_transportation_chat_id' => $booking->status == 2 ? ($airTransportation ? $airTransportation->sendbird_chat_id : "") : "",
                    'air_transportation_phone_no' => $booking->status == 2 ? ($airTransportation ? $airTransportation->mobile : "") : "",
                    'booking_price' => ($booking->status != 1 && $booking->status != 4) ? $booking->booking_price : "0",
                    'booking_status' => $booking->status
                ];

                // return $data;
            });

            return response()->json([
                'message' => __('api.airtransport.booking_fetch'),
                'data' => $finalData,
                'pagination' => [
                    'current_page' => $bookings->currentPage(),
                    'per_page' => $bookings->perPage(),
                    'total' => $bookings->total(),
                    'total_page' => $bookings->lastPage()
                ],
            ]);
        } catch (\Exception $e) {
            // Log the error if needed
            // Log::error($e->getMessage());

            return response()->json([
                'message' => __('api.airtransport.fetch_error'),
            ], 500);
        }
    }

    public function getBookingDetails(Request $request)
    {
        try {
            $booking = AirTransportationBooking::findOrFail($request->id);
            $airTransportation = $booking->air_transportations;
            $count= Review::where('reviewable_id',$request->id)->where('reviewable_type','App\Models\AirTransportationBooking')->count();
            $data = [
                'id' => $booking->id,
                'booking_no' => '#'.$booking->booking_no,
                'booking_date_time' => formatDate($booking->created_at),
                'pickup_location' => $booking->pickup_location,
                'dropoff_location' => $booking->dropoff_location,
                'adult' => $booking->adult,
                'child' => $booking->child,
                'flying_type' => getFlyingTypeText($booking->flying_type),
                'trip_type' => $booking->trip_type,
                'departure_date_time' => quoteDate($booking->departure_date_time),
                'arrival_date_time' => $booking->arrival_date_time == null ? '' : quoteDate($booking->arrival_date_time),
                'comment' => $booking->comment,
                'air_transportation_image' => $airTransportation ? ($airTransportation->profile_image == null ? '' : $airTransportation->profile_image) : '',
                'air_transportation_name' => $airTransportation ? $airTransportation->getFullName() : 'Not Applicable',
                'booking_price' => ($booking->status != 1 && $booking->status != 4) ? $booking->booking_price : "0",
                'booking_status' => $booking->status,
                'is_rated' => $count == 1 ? 1 : 0,
                'cancel_reason' => $booking->status == 4 ? $booking->cancel_reason : '',
            ];

            return response()->json([
                'message' => __('api.airtransport.booking_details'),
                'data' => $data
            ]);
        } catch (\Exception $exception) {
            // Log the error if needed
            // Log::error($exception->getMessage());

            return response()->json([
                'message' => __('api.airtransport.detail_error'),
                // 'error' => $exception->getMessage()
            ], 500);
        }
    }

    public function addQuote(Request $request)
    {
        try {
            $booking = AirTransportationBooking::findOrFail($request->air_transportation_booking_id);
            $res = Quote::create([
                'air_transportation_booking_id' => $request->air_transportation_booking_id,
                'air_transportation_id' => $request->air_transportation_id,
                'company_quote_price' => $request->company_quote_price,
                'quote_date_time' => $request->quote_date_time,
                'quote_status' => 2
            ]);
            
            Notification::create([
                'notification_title' => 'Quote received',
                'notification_desc' => 'company given a quotation',
                'notification_object_id' => $res->id,
                'notification_object_type' => 8,
                'sender_id' => $request->air_transportation_id,
                'receiver_id' => $booking->user_id,
                'sender_type' => 'airtransport',
                'receiver_type' => 'customer',
                'type' => 3
            ]);
        } catch (\Exception $exception) {
            // Log the error if needed
            // Log::error($exception->getMessage());

            return response()->json([
                // 'message' => __('api.airtransport.detail_error'),
                'error' => $exception->getMessage()
            ], 500);
        }
        echo "inserted";
    }

    public function completeBooking(Request $request)
    {
        AirTransportationBooking::where('id', $request->id)->update(['status' => 3]);

        echo "inserted";
    }
}
