<?php

namespace App\Exceptions;

use Throwable;
use Illuminate\Support\Arr;
use Illuminate\Auth\AuthenticationException;
use Symfony\Component\HttpFoundation\Response;
use Illuminate\Foundation\Exceptions\Handler as ExceptionHandler;

class Handler extends ExceptionHandler
{
    /**
     * A list of the exception types that are not reported.
     *
     * @var array<int, class-string<Throwable>>
     */
    protected $dontReport = [
        //
    ];

    /**
     * A list of the inputs that are never flashed for validation exceptions.
     *
     * @var array<int, string>
     */
    protected $dontFlash = [
        'current_password',
        'password',
        'password_confirmation',
    ];

    /**
     * Register the exception handling callbacks for the application.
     *
     * @return void
     */
    public function register()
    {
        $this->reportable(function (Throwable $e) {
            //
        });
    }

    /**
     * Render an exception into an HTTP response.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Throwable  $exception
     * @return \Symfony\Component\HttpFoundation\Response
     *
     * @throws \Throwable
     */
    public function render($request, Throwable $exception)
    {
      
        if($this->isAuthenticationException($exception)) {

            if ($request->wantsJson()) {
                return response()->json([
                    'message'=> __('auth.invalid_token')
                ], Response::HTTP_UNAUTHORIZED);
            }
            
            return redirect()->to(
                $this->redirectPath($exception)
            );
        }

        if ($this->isAppUnderMaintainance($exception)) {
            if ($request->wantsJson()) {
                return response()->json([
                        'message'=> __('auth.under_maintenance')
                ], Response::HTTP_SERVICE_UNAVAILABLE);
            }
        }

        
        if ($exception instanceof \Illuminate\Session\TokenMismatchException) {
           
            session()->flash('error', 'The page expired, please try again.');
            return back()->with([
                'message' => 'The page expired, please try again.',
            ]);
        }
        return parent::render($request, $exception);
    }

    public function isAuthenticationException($exception)
    {
        return $exception instanceof AuthenticationException;
    }
    
    public function redirectPath($exception)
    {
        return match(Arr::get($exception->guards(), 0)) {
            'sanctum' => '',
            'admin' => route('admin.login'),
            default => route('airtransport.login')
        };
    }

    public function isAppUnderMaintainance($exception)
    {
        return method_exists($exception, 'getStatusCode') 
                && $exception->getStatusCode() == Response::HTTP_SERVICE_UNAVAILABLE;
    }
}
